const Admin = require('../../Model/Admin/Admin_Model');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const User = require("../../Model/User_Model")
const adminLogin = async (req, res) => {
  try {
    const { email, password } = req.body;
    
    if (email !== 'novellhub19@gmail.com') {
      return res.status(401).json({ error: 'Invalid admin credentials' });
    }

    const admin = await Admin.findOne({ email });
    if (!admin) {
      return res.status(404).json({ error: 'Admin not found' });
    }

    const isValidPassword = await bcrypt.compare(password, admin.password);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid password' });
    }

    const token = jwt.sign({ 
      adminId: admin._id,
      role: 'admin' 
    }, process.env.JWT_SECRET);

    res.status(200).json({
      success: true,
      adminToken: token,
      admin: {
        id: admin._id,
        email: admin.email
      },
      redirectUrl: '/admin/dashboard' // Add redirect URL in response
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

  

const registerAdmin = async (req, res) => {
  try {
    const { email, password } = req.body;
    const hashedPassword = await bcrypt.hash(password, 10);

    const admin = new Admin({
      email,
      password: hashedPassword
    });

    await admin.save();
    res.status(201).json({
      success: true,
      message: 'Admin registered successfully'
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const checkUserExists = async (req, res) => {
  try {
    const { identifier } = req.query;
    
    if (!identifier) {
      return res.status(400).json({ error: "Email or username is required" });
    }
    
    // Check if the identifier is an email or username
    const query = identifier.includes('@') 
      ? { email: identifier.toLowerCase() } 
      : { username: identifier };
    
    // Find the user but don't return sensitive information
    const user = await User.findOne(query).select('username email profileImage');
    
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.status(200).json({
      success: true,
      user
    });
  } catch (error) {
    console.error('Error checking user existence:', error);
    res.status(500).json({ error: error.message });
  }
};

// Admin reset user password - Alternative approach
const adminResetUserPassword = async (req, res) => {
  try {
    const { identifier, newPassword } = req.body;
    
    console.log('Password reset request received:', { 
      identifier, 
      newPasswordLength: newPassword?.length,
      timestamp: new Date().toISOString()
    });
    
    // Validate input
    if (!identifier || !newPassword) {
      console.log('Validation failed: Missing identifier or password');
      return res.status(400).json({ 
        error: "User identifier and new password are required" 
      });
    }
    
    if (newPassword.length < 6) {
      console.log('Validation failed: Password too short');
      return res.status(400).json({ 
        error: "Password must be at least 6 characters long" 
      });
    }
    
    // Check if the identifier is an email or username
    const query = identifier.includes('@') 
      ? { email: identifier.toLowerCase().trim() } 
      : { username: identifier.trim() };
    
    console.log('Searching for user with query:', query);
    
    // Find the user
    const user = await User.findOne(query);
    
    if (!user) {
      console.log('User not found with query:', query);
      return res.status(404).json({ 
        error: "User not found with the provided email or username" 
      });
    }
    
    console.log('User found successfully:', { 
      id: user._id, 
      username: user.username, 
      email: user.email 
    });

    // Hash the new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);
    
    // Use direct MongoDB update to bypass Mongoose validation
    const updateResult = await User.collection.updateOne(
      { _id: user._id },
      { 
        $set: { 
          password: hashedPassword
        },
        $pull: {
          notifications: {
            $or: [
              { contentId: { $exists: false } },
              { contentId: null },
              { contentModel: { $exists: false } },
              { contentModel: null }
            ]
          }
        }
      }
    );
    
    console.log('Password update result:', updateResult);
    
    if (updateResult.modifiedCount === 0) {
      throw new Error('Failed to update password');
    }
    
    console.log('Password updated successfully for user:', user.username);
    
    res.status(200).json({
      success: true,
      message: `Password for user "${user.username}" has been reset successfully`
    });
    
  } catch (error) {
    console.error('Error resetting user password:', error);
    console.error('Error stack:', error.stack);
    
    res.status(500).json({ 
      error: "Failed to reset password",
      message: error.message,
      details: process.env.NODE_ENV === 'development' ? error.stack : undefined
    });
  }
};





module.exports = {
  adminLogin,
  registerAdmin,
  checkUserExists,
  adminResetUserPassword
};
