// Admin Controller
const User = require("../../Model/User_Model");
const Story = require("../../Model/Story_Model");
const Series = require("../../Model/Series_Model");

// Find this function in your Admin_Controller.js
const getAllWriters = async (req, res) => {
  try {
    // Update this query to include phoneNumber
    const writers = await User.find()
      .select('username email phoneNumber writingStats lastActive') // Add phoneNumber here
      .sort({ createdAt: -1 });
    
    res.status(200).json({
      success: true,
      writers
    });
  } catch (error) {
    console.error("Error fetching writers:", error);
    res.status(500).json({ error: error.message });
  }
};


const getAdminStats = async (req, res) => {
  try {
    // Get unique authors from both stories and series
    const storyWriters = await Story.distinct("author", {
      status: "published",
    });
    const seriesWriters = await Series.distinct("author", {
      status: { $ne: "draft" },
    });

    // Combine unique writers
    const activeWriters = [...new Set([...storyWriters, ...seriesWriters])]
      .length;

    // Calculate total words and payments
    const writers = await User.find({ role: "user" });
    const totalWords = writers.reduce(
      (sum, writer) => sum + (writer.writingStats?.totalWordCount || 0),
      0
    );

    const totalPayments = writers.reduce((total, writer) => {
      const writerPayments =
        writer.paymentHistory?.reduce(
          (sum, payment) => sum + (payment.amount || 0),
          0
        ) || 0;
      return total + writerPayments;
    }, 0);

    res.status(200).json({
      success: true,
      stats: {
        totalWriters: writers.length,
        activeWriters,
        totalWords,
        totalEarnings: totalPayments,
      },
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getWriterStats = async (req, res) => {
  try {
    const writer = await User.findById(req.params.id);
    if (!writer) {
      return res.status(404).json({ error: "Writer not found" });
    }

    const dateKey = req.query.date;
    const dailyStats = writer.writingStats?.dailyWordCount?.get(dateKey);

    if (dailyStats && (dailyStats.count > 0 || dailyStats.earnings > 0)) {
      const earnings = dailyStats.count * 0.1;

      res.status(200).json({
        success: true,
        stats: {
          date: dateKey,
          wordCount: dailyStats.count,
          earnings: earnings,
        },
      });
    } else {
      res.status(200).json({
        success: true,
        stats: null,
      });
    }
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getWriterAccountDetails = async (req, res) => {
  try {
    const writer = await User.findById(req.params.id).select("accountDetails");
    if (!writer) {
      return res
        .status(404)
        .json({ success: false, error: "Writer not found" });
    }
    res.json({ success: true, accountDetails: writer.accountDetails });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

const recordWriterPayment = async (req, res) => {
  try {
    const { writerId, amount, paidFrom, paidTo } = req.body;
    const writer = await User.findById(writerId);

    if (!writer) {
      return res.status(404).json({ error: "Writer not found" });
    }

    writer.paymentHistory.push({
      amount,
      paidFrom: new Date(paidFrom),
      paidTo: new Date(paidTo),
      paidOn: new Date(),
    });

    await writer.save();

    res.status(200).json({
      success: true,
      message: "Payment recorded successfully",
      paymentHistory: writer.paymentHistory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getWriterPayments = async (req, res) => {
  try {
    const writer = await User.findById(req.params.id).select("paymentHistory");
    res.json({ success: true, paymentHistory: writer.paymentHistory });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updatePayment = async (req, res) => {
  try {
    const { writerId, paymentId, amount, paidFrom, paidTo } = req.body;
    const writer = await User.findById(writerId);

    const payment = writer.paymentHistory.id(paymentId);
    if (!payment) {
      return res.status(404).json({ error: "Payment not found" });
    }

    payment.amount = amount;
    payment.paidFrom = paidFrom;
    payment.paidTo = paidTo;
    await writer.save();

    res.status(200).json({
      success: true,
      message: "Payment updated successfully",
      paymentHistory: writer.paymentHistory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deletePayment = async (req, res) => {
  try {
    const { writerId, paymentId } = req.params;
    const writer = await User.findById(writerId);

    writer.paymentHistory = writer.paymentHistory.filter(
      (payment) => payment._id.toString() !== paymentId
    );
    await writer.save();

    res.status(200).json({
      success: true,
      message: "Payment deleted successfully",
      paymentHistory: writer.paymentHistory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getWriterDetails = async (req, res) => {
  try {
    const writer = await User.findById(req.params.writerId)
      .select("-password")
      .populate("followers")
      .populate("following");

    if (!writer) {
      return res
        .status(404)
        .json({ success: false, message: "Writer not found" });
    }

    // Continuing from getWriterDetails...

    res.status(200).json({
      success: true,
      user: writer,
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

const getWriterStories = async (req, res) => {
  try {
    const stories = await Story.find({
      author: req.params.writerId,
      status: "published",
    }).populate("author", "username");

    res.status(200).json({
      success: true,
      stories,
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

const getWriterSeries = async (req, res) => {
  try {
    const series = await Series.find({
      author: req.params.writerId,
      status: { $ne: "draft" },
    }).populate("author", "username");

    res.status(200).json({
      success: true,
      series,
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

const deleteStory = async (req, res) => {
  try {
    const story = await Story.findById(req.params.storyId);
    if (!story) {
      return res.status(404).json({ error: "Story not found" });
    }

    // Calculate word count and earnings to subtract
    const wordCount = story.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;
    const earnings = wordCount * 0.1;
    const dateKey = story.createdAt.toISOString().split("T")[0];

    // Update user stats
    await User.findByIdAndUpdate(story.author, {
      $inc: {
        "writingStats.totalWordCount": -wordCount,
        "writingStats.totalEarnings": -earnings,
        [`writingStats.dailyWordCount.${dateKey}.count`]: -wordCount,
        [`writingStats.dailyWordCount.${dateKey}.earnings`]: -earnings,
      },
    });

    // Delete story and related notifications
    await Promise.all([
      Story.findByIdAndDelete(req.params.storyId),
      User.updateMany(
        { "notifications.contentId": story._id },
        { $pull: { notifications: { contentId: story._id } } }
      ),
    ]);

    res.status(200).json({
      success: true,
      message: "Story and related data deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteSeries = async (req, res) => {
  try {
    const series = await Series.findById(req.params.seriesId);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    // Calculate total words and earnings from all episodes
    const totalWords = series.episodes.reduce((sum, episode) => {
      return (
        sum +
        episode.content.split(/\s+/).filter((word) => word.length > 0).length
      );
    }, 0);
    const totalEarnings = totalWords * 0.1;
    const dateKey = series.createdAt.toISOString().split("T")[0];

    // Update user stats
    await User.findByIdAndUpdate(series.author, {
      $inc: {
        "writingStats.totalWordCount": -totalWords,
        "writingStats.totalEarnings": -totalEarnings,
        [`writingStats.dailyWordCount.${dateKey}.count`]: -totalWords,
        [`writingStats.dailyWordCount.${dateKey}.earnings`]: -totalEarnings,
      },
    });

    // Delete series and related notifications
    await Promise.all([
      Series.findByIdAndDelete(req.params.seriesId),
      User.updateMany(
        { "notifications.contentId": series._id },
        { $pull: { notifications: { contentId: series._id } } }
      ),
    ]);

    res.status(200).json({
      success: true,
      message: "Series and related data deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateStory = async (req, res) => {
  try {
    const story = await Story.findByIdAndUpdate(
      req.params.storyId,
      { ...req.body },
      { new: true }
    );
    res.status(200).json({
      success: true,
      story,
      message: "Story updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateSeries = async (req, res) => {
  try {
    const series = await Series.findByIdAndUpdate(
      req.params.seriesId,
      { ...req.body },
      { new: true }
    );
    res.status(200).json({
      success: true,
      series,
      message: "Series updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getEnhancedAdminStats = async (req, res) => {
  try {
    // Get current date and date for yesterday
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);

    // Format dates for comparison
    const todayStart = new Date(today.setHours(0, 0, 0, 0));
    const yesterdayStart = new Date(yesterday.setHours(0, 0, 0, 0));

    // Get total users count
    const totalUsers = await User.countDocuments();

    // Get daily active users (users active in the last 24 hours)
    const dailyActiveUsers = await User.countDocuments({
      lastActive: { $gte: yesterdayStart },
    });

    // Get total writers (users who have published content)
    const storyWriters = await Story.distinct("author", {
      status: "published",
    });

    const seriesWriters = await Series.distinct("author", {
      status: { $ne: "draft" },
    });

    // Combine unique writers from both stories and series
    const allWriters = [...new Set([...storyWriters, ...seriesWriters])];
    const totalWriters = allWriters.length;

    // Get active writers (writers who published in the last 7 days)
    const sevenDaysAgo = new Date();
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

    const recentStoryWriters = await Story.distinct("author", {
      status: "published",
      updatedAt: { $gte: sevenDaysAgo },
    });

    const recentSeriesWriters = await Series.distinct("author", {
      status: { $ne: "draft" },
      updatedAt: { $gte: sevenDaysAgo },
    });

    const activeWriters = [
      ...new Set([...recentStoryWriters, ...recentSeriesWriters]),
    ].length;

    // Calculate total words written
    const writers = await User.find({ role: "user" });
    const totalWords = writers.reduce(
      (sum, writer) => sum + (writer.writingStats?.totalWordCount || 0),
      0
    );

    // Get words written today
    const todayWords = writers.reduce((sum, writer) => {
      const today = new Date().toISOString().split("T")[0];
      const dailyCount =
        writer.writingStats?.dailyWordCount?.get(today)?.count || 0;
      return sum + dailyCount;
    }, 0);

    res.status(200).json({
      success: true,
      stats: {
        totalUsers,
        dailyActiveUsers,
        totalWriters,
        activeWriters,
        totalWords,
        todayWords,
      },
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getActivityTrend = async (req, res) => {
  try {
    const activityData = [];
    const today = new Date();

    // Get data for the last 7 days
    for (let i = 6; i >= 0; i--) {
      const date = new Date();
      date.setDate(today.getDate() - i);

      // Start of the day
      const dayStart = new Date(date);
      dayStart.setHours(0, 0, 0, 0);

      // End of the day
      const dayEnd = new Date(date);
      dayEnd.setHours(23, 59, 59, 999);

      // Count active users for this day
      const activeUsers = await User.countDocuments({
        lastActive: { $gte: dayStart, $lte: dayEnd },
      });

      // Count active writers (users who published content) for this day
      const storyWriters = await Story.distinct("author", {
        status: "published",
        updatedAt: { $gte: dayStart, $lte: dayEnd },
      });

      const seriesWriters = await Series.distinct("author", {
        status: { $ne: "draft" },
        updatedAt: { $gte: dayStart, $lte: dayEnd },
      });

      const activeWriters = [...new Set([...storyWriters, ...seriesWriters])]
        .length;

      // Format date as short weekday
      const dateStr = date.toLocaleDateString("en-US", { weekday: "short" });

      activityData.push({
        day: dateStr,
        users: activeUsers,
        writers: activeWriters,
      });
    }

    res.status(200).json({
      success: true,
      activityTrend: activityData,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
module.exports = {
  getAllWriters,
  getAdminStats,
  getWriterStats,
  getWriterAccountDetails,
  recordWriterPayment,
  getWriterPayments,
  updatePayment,
  deletePayment,
  getWriterDetails,
  getWriterStories,
  getWriterSeries,
  deleteStory,
  deleteSeries,
  updateStory,
  updateSeries,
  getEnhancedAdminStats,
  getActivityTrend,
};
