const Category = require("../../Model/Admin/Category_admin_Model");
const fs = require("fs");
const path = require("path");

// Add this function to update category display order
const updateCategoryOrder = async (req, res) => {
  try {
    const { categoryOrders } = req.body;
    
    if (!Array.isArray(categoryOrders)) {
      return res.status(400).json({ 
        success: false, 
        error: "categoryOrders must be an array of {id, order} objects" 
      });
    }
    
    // Process each category order update
    const updatePromises = categoryOrders.map(item => {
      if (!item.id || typeof item.order !== 'number') {
        return Promise.reject(new Error("Each item must have id and order"));
      }
      
      return Category.findByIdAndUpdate(
        item.id,
        { displayOrder: item.order },
        { new: true }
      );
    });
    
    await Promise.all(updatePromises);
    
    // Fetch updated categories
    const updatedCategories = await Category.find().sort({ displayOrder: 1, title: 1 });
    
    res.status(200).json({
      success: true,
      categories: updatedCategories,
      message: "Category order updated successfully"
    });
  } catch (error) {
    console.error("Error updating category order:", error);
    res.status(500).json({ error: error.message });
  }
};

// Update the addCategory function to handle displayOrder
const addCategory = async (req, res) => {
  try {
    const { title, isActive, displayOrder, showOnHome } = req.body;
    
    // Check if category already exists
    const existingCategory = await Category.findOne({ title });
    if (existingCategory) {
      return res.status(400).json({ 
        success: false,
        error: "Category with this title already exists" 
      });
    }
    
    // Get the highest display order if not provided
    let order = displayOrder;
    if (order === undefined) {
      const highestOrder = await Category.findOne().sort('-displayOrder');
      order = highestOrder ? highestOrder.displayOrder + 1 : 1;
    }
    
    // Create new category
    const category = new Category({
      title,
      isActive: isActive !== undefined ? isActive : true,
      showOnHome: showOnHome !== undefined ? showOnHome : false,
      displayOrder: order,
      coverImage: req.file ? `/uploads/categories/${req.file.filename}` : "",
      createdBy: req.userId
    });
      
    const savedCategory = await category.save();
    
    res.status(201).json({
      success: true,
      category: savedCategory,
      message: "Category added successfully"
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update the updateCategory function to handle displayOrder
const updateCategory = async (req, res) => {
  try {
    const { title, isActive, displayOrder, showOnHome } = req.body;
    const categoryId = req.params.id;
    
    const category = await Category.findById(categoryId);
    if (!category) {
      return res.status(404).json({ 
        success: false,
        error: "Category not found" 
      });
    }
    
    // Check if new title already exists (but not for this category)
    if (title && title !== category.title) {
      const existingCategory = await Category.findOne({ title });
      if (existingCategory) {
        return res.status(400).json({ 
          success: false,
          error: "Category with this title already exists" 
        });
      }
    }
    
    // Update fields
    if (title) category.title = title;
    if (isActive !== undefined) category.isActive = isActive;
    if (showOnHome !== undefined) category.showOnHome = showOnHome;
    if (displayOrder !== undefined) category.displayOrder = displayOrder;
    
    // Handle cover image update
    if (req.file) {
      // Delete old image if it exists
      if (category.coverImage) {
        const oldImagePath = path.join(__dirname, '../../', category.coverImage);
        if (fs.existsSync(oldImagePath)) {
          fs.unlinkSync(oldImagePath);
        }
      }
      
      category.coverImage = `/uploads/categories/${req.file.filename}`;
    }
    
    const updatedCategory = await category.save();
    
    res.status(200).json({
      success: true,
      category: updatedCategory,
      message: "Category updated successfully"
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update the getAllCategories function to sort by displayOrder
const getAllCategories = async (req, res) => {
  try {
    const categories = await Category.find().sort({ displayOrder: 1, title: 1 });
    res.status(200).json({
      success: true,
      categories,
      message: "Categories fetched successfully"
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};



// Delete a category
const deleteCategory = async (req, res) => {
  try {
    const categoryId = req.params.id;
    
    const category = await Category.findById(categoryId);
    if (!category) {
      return res.status(404).json({ 
        success: false,
        error: "Category not found" 
      });
    }
    
    // Delete cover image if it exists
    if (category.coverImage) {
      const imagePath = path.join(__dirname, '../../', category.coverImage);
      if (fs.existsSync(imagePath)) {
        fs.unlinkSync(imagePath);
      }
    }
    
    await Category.findByIdAndDelete(categoryId);
    
    res.status(200).json({
      success: true,
      message: "Category deleted successfully"
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Toggle category status (active/inactive)
const toggleCategoryStatus = async (req, res) => {
  try {
    const categoryId = req.params.id;
    
    const category = await Category.findById(categoryId);
    if (!category) {
      return res.status(404).json({ 
        success: false,
        error: "Category not found" 
      });
    }
    
    category.isActive = !category.isActive;
    await category.save();

    res.status(200).json({
      success: true,
      isActive: category.isActive,
      message: `Category ${category.isActive ? 'activated' : 'deactivated'} successfully`
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const toggleCategoryHomeStatus = async (req, res) => {
  try {
    const { id } = req.params;
    
    const category = await Category.findById(id);
    if (!category) {
      return res.status(404).json({ error: 'Category not found' });
    }
    
    // Toggle the showOnHome status
    category.showOnHome = !category.showOnHome;
    await category.save();
    
    res.status(200).json({
      success: true,
      category,
      message: `Category ${category.showOnHome ? 'added to' : 'removed from'} home screen successfully`
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  updateCategoryOrder,
  getAllCategories,
  addCategory,
  updateCategory,
  deleteCategory,
  toggleCategoryStatus,
  toggleCategoryHomeStatus
};
