const UserTraffic = require('../../Model/Admin/UserTraffic_Model');

// Get user traffic data for admin dashboard
const getUserTraffic = async (req, res) => {
  try {
    const { date } = req.query;
    
    let query = {};
    if (date) {
      const startDate = new Date(date);
      startDate.setHours(0, 0, 0, 0);
      
      const endDate = new Date(date);
      endDate.setHours(23, 59, 59, 999);
      
      query.entryTime = { $gte: startDate, $lte: endDate };
    }
    
    const trafficData = await UserTraffic.find(query)
      .sort({ entryTime: -1 })
      .limit(100);
    
    // Calculate stats
    const totalVisits = await UserTraffic.countDocuments();
    const uniqueIPs = await UserTraffic.distinct('ipAddress');
    const loggedInVisits = await UserTraffic.countDocuments({ username: { $ne: null } });
    
    res.json({
      success: true,
      trafficData,
      stats: {
        totalVisits,
        uniqueUsers: uniqueIPs.length,
        loggedInVisits,
        anonymousVisits: totalVisits - loggedInVisits
      }
    });
  } catch (error) {
    console.error('Error fetching user traffic:', error);
    res.status(500).json({ success: false, message: 'Server error' });
  }
};

// Record user entry
const recordEntry = async (req, res) => {
  try {
    const { userId, username, sessionId } = req.body;
    const ipAddress = req.ip || req.headers['x-forwarded-for'] || 'Unknown';
    const userAgent = req.headers['user-agent'] || 'Unknown';

    // Check if there's already an active session with this sessionId
    const existingSession = await UserTraffic.findOne({ 
      sessionId, 
      exitTime: null 
    });

    if (existingSession) {
      // Session already exists and is active, no need to create a new one
      return res.json({ success: true, message: 'Session already active' });
    }

    const newTrafficRecord = new UserTraffic({
      userId,
      username,
      ipAddress,
      userAgent,
      sessionId,
      entryTime: new Date()
    });

    await newTrafficRecord.save();
    res.json({ success: true, message: 'Entry recorded successfully' });
  } catch (error) {
    console.error('Error recording user entry:', error);
    res.status(500).json({ success: false, message: 'Server error' });
  }
};

// Record user exit
const recordExit = async (req, res) => {
  try {
    const { sessionId } = req.body;
    
    const result = await UserTraffic.findOneAndUpdate(
      { sessionId, exitTime: null },
      { exitTime: new Date() },
      { new: true }
    );
    
    if (result) {
      res.json({ success: true, message: 'Exit recorded successfully' });
    } else {
      res.json({ success: false, message: 'No active session found' });
    }
  } catch (error) {
    console.error('Error recording user exit:', error);
    res.status(500).json({ success: false, message: 'Server error' });
  }
};

module.exports = {
  getUserTraffic,
  recordEntry,
  recordExit
};
