const Contact = require("../Model/Contact_Model");
const User = require("../Model/User_Model");
const nodemailer = require('nodemailer');
const { createContactReplyNotification } = require('../utils/notifications');

// Configure nodemailer
const transporter = nodemailer.createTransport({
  service: 'gmail',
  auth: {
    user: process.env.EMAIL_USER || 'novellhub19@gmail.com',
    pass: process.env.EMAIL_PASSWORD
  }
});

// Helper function to send emails
const sendEmail = async (options) => {
  const mailOptions = {
    from: 'NovellHub <novellhub19@gmail.com>',
    to: options.to,
    subject: options.subject,
    html: options.html
  };

  return transporter.sendMail(mailOptions);
};

// Submit a new contact form
const submitContactForm = async (req, res) => {
  try {
    const { name, email, subject, message, userId } = req.body;

    // Validate input
    if (!name || !email || !subject || !message) {
      return res.status(400).json({ 
        success: false, 
        error: "All fields are required" 
      });
    }

    // Create new contact entry
    const newContact = new Contact({
      name,
      email,
      subject,
      message,
      userId: userId || null // Store user ID if provided
    });

    await newContact.save();

    // Send confirmation email to user
    try {
      await sendEmail({
        to: email,
        subject: "Thank you for contacting NovellHub",
        html: `
          <h2>Thank you for reaching out to NovellHub!</h2>
          <p>Dear ${name},</p>
          <p>We have received your message regarding "${subject}". Our team will review it and get back to you as soon as possible.</p>
          <p>For your reference, here's a copy of your message:</p>
          <div style="background-color: #f5f5f5; padding: 15px; border-radius: 5px; margin: 15px 0;">
            <p>${message}</p>
          </div>
          <p>Best regards,</p>
          <p>The NovellHub Team</p>
        `
      });
    } catch (emailError) {
      console.error("Error sending confirmation email:", emailError);
      // Continue even if email fails
    }

    // Send notification email to admin
    try {
      await sendEmail({
        to: process.env.ADMIN_EMAIL || 'novellhub19@gmail.com',
        subject: "New Contact Form Submission",
        html: `
          <h2>New Contact Form Submission</h2>
          <p><strong>Name:</strong> ${name}</p>
          <p><strong>Email:</strong> ${email}</p>
          <p><strong>Subject:</strong> ${subject}</p>
          <p><strong>Message:</strong></p>
          <div style="background-color: #f5f5f5; padding: 15px; border-radius: 5px; margin: 15px 0;">
            <p>${message}</p>
          </div>
          <p><a href="${process.env.ADMIN_URL || 'http://localhost:3000'}/admin/contacts">View in Admin Panel</a></p>
        `
      });
    } catch (emailError) {
      console.error("Error sending admin notification email:", emailError);
      // Continue even if email fails
    }

    res.status(201).json({
      success: true,
      message: "Your message has been sent successfully. We'll get back to you soon!",
      contactId: newContact._id
    });
  } catch (error) {
    console.error("Error submitting contact form:", error);
    res.status(500).json({ 
      success: false, 
      error: "Failed to submit your message. Please try again." 
    });
  }
};

// Get all contact submissions (admin only)
const getAllContacts = async (req, res) => {
  try {
    const contacts = await Contact.find().sort({ createdAt: -1 });
    
    res.status(200).json({
      success: true,
      count: contacts.length,
      contacts
    });
  } catch (error) {
    console.error("Error fetching contacts:", error);
    res.status(500).json({ 
      success: false, 
      error: "Failed to fetch contact submissions" 
    });
  }
};

// Get a single contact by ID (admin only)
const getContactById = async (req, res) => {
  try {
    const contact = await Contact.findById(req.params.id);
    
    if (!contact) {
      return res.status(404).json({
        success: false,
        error: "Contact submission not found"
      });
    }
    
    res.status(200).json({
      success: true,
      contact
    });
  } catch (error) {
    console.error("Error fetching contact:", error);
    res.status(500).json({ 
      success: false, 
      error: "Failed to fetch contact submission" 
    });
  }
};

// Update contact status (admin only)
const updateContactStatus = async (req, res) => {
  try {
    const { status, adminNotes } = req.body;
    
    if (!status || !['unread', 'read', 'replied'].includes(status)) {
      return res.status(400).json({
        success: false,
        error: "Invalid status"
      });
    }
    
    const contact = await Contact.findByIdAndUpdate(
      req.params.id,
      { 
        status,
        ...(adminNotes && { adminNotes })
      },
      { new: true }
    );
    
    if (!contact) {
      return res.status(404).json({
        success: false,
        error: "Contact submission not found"
      });
    }
    
    res.status(200).json({
      success: true,
      message: "Contact status updated successfully",
      contact
    });
  } catch (error) {
    console.error("Error updating contact status:", error);
    res.status(500).json({ 
      success: false, 
      error: "Failed to update contact status" 
    });
  }
};

// Update the replyToContact function
const replyToContact = async (req, res) => {
    try {
      const { replyMessage } = req.body;
      
      if (!replyMessage || !replyMessage.trim()) {
        return res.status(400).json({
          success: false,
          error: "Reply message is required"
        });
      }
      
      const contact = await Contact.findById(req.params.id);
      
      if (!contact) {
        return res.status(404).json({
          success: false,
          error: "Contact submission not found"
        });
      }
      
      console.log("Contact found:", {
        id: contact._id,
        name: contact.name,
        email: contact.email,
        userId: contact.userId,
        hasUserId: !!contact.userId
      });
      
      // Update contact status to replied
      contact.status = 'replied';
      contact.adminNotes = contact.adminNotes || '';
      contact.adminNotes += `\n[${new Date().toISOString()}] Reply sent: ${replyMessage}`;
      
      // Add reply to replies array
      contact.replies = contact.replies || [];
      contact.replies.push({
        message: replyMessage,
        createdAt: new Date()
      });
      
      await contact.save();
      
      // Send email reply to the user
      try {
        await sendEmail({
          to: contact.email,
          subject: `Re: ${contact.subject} - NovellHub Support`,
          html: `
            <h2>Response from NovellHub Support</h2>
            <p>Dear ${contact.name},</p>
            <p>Thank you for contacting NovellHub. Here is our response to your inquiry:</p>
            <div style="background-color: #f5f5f5; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <p>${replyMessage}</p>
            </div>
            <p>Your original message:</p>
            <div style="border-left: 4px solid #ccc; padding-left: 15px; margin: 15px 0;">
              <p><strong>Subject:</strong> ${contact.subject}</p>
              <p>${contact.message}</p>
            </div>
            <p>If you have any further questions, please don't hesitate to contact us again.</p>
            <p>Best regards,</p>
            <p>The NovellHub Team</p>
          `
        });
      } catch (emailError) {
        console.error("Error sending reply email:", emailError);
        // Continue even if email fails
      }
      
      // Create notification for the user if userId exists
      if (contact.userId) {
        try {
          console.log("Creating notification for user:", contact.userId);
          
          // Create notification directly in the user document
          const notificationData = {
            type: "contact_reply",
            message: `We've responded to your message: "${contact.subject}"`,
            fromUser: null,
            contentId: contact._id,
            contentModel: "Contact",
            createdAt: new Date(),
            isRead: false,
            additionalData: {
              reply: replyMessage,
              originalSubject: contact.subject
            }
          };
          
          console.log("Adding notification with data:", JSON.stringify(notificationData));
          
          // Use findByIdAndUpdate to ensure proper data structure
          const result = await User.findByIdAndUpdate(
            contact.userId,
            { 
              $push: { 
                notifications: notificationData
              } 
            },
            { new: true }
          );
          
          // Verify the notification was added correctly
          const updatedUser = await User.findById(contact.userId);
          const addedNotification = updatedUser.notifications.find(
            n => n.type === 'contact_reply' && 
            n.contentId.toString() === contact._id.toString() &&
            new Date(n.createdAt).getTime() > Date.now() - 60000 // Added in the last minute
          );
          
          console.log("Added notification:", addedNotification);
          console.log("Has additionalData:", !!addedNotification?.additionalData);
        } catch (notifError) {
          console.error("Error creating notification:", notifError);
        }
      } else {
        console.log("No userId found for this contact, skipping notification");
      }
      
      res.status(200).json({
        success: true,
        message: "Reply sent successfully",
        contact
      });
    } catch (error) {
      console.error("Error replying to contact:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to send reply" 
      });
    }
  };
  
  
  
  // Delete a contact (admin only)
  const deleteContact = async (req, res) => {
    try {
      const contact = await Contact.findByIdAndDelete(req.params.id);
      
      if (!contact) {
        return res.status(404).json({
          success: false,
          error: "Contact submission not found"
        });
      }
      
      res.status(200).json({
        success: true,
        message: "Contact submission deleted successfully"
      });
    } catch (error) {
      console.error("Error deleting contact:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to delete contact submission" 
      });
    }
  };
  
  // Get contact statistics (admin only)
  const getContactStats = async (req, res) => {
    try {
      const totalContacts = await Contact.countDocuments();
      const unreadContacts = await Contact.countDocuments({ status: 'unread' });
      const readContacts = await Contact.countDocuments({ status: 'read' });
      const repliedContacts = await Contact.countDocuments({ status: 'replied' });
      
      // Get contacts from the last 7 days
      const lastWeekContacts = await Contact.countDocuments({
        createdAt: { $gte: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000) }
      });
      
      res.status(200).json({
        success: true,
        stats: {
          total: totalContacts,
          unread: unreadContacts,
          read: readContacts,
          replied: repliedContacts,
          lastWeek: lastWeekContacts
        }
      });
    } catch (error) {
      console.error("Error fetching contact stats:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch contact statistics" 
      });
    }
  };
  
const contactMessage = async (req,res) =>{
    try {
        const contact = await Contact.findById(req.params.id);
        
        if (!contact) {
          return res.status(404).json({
            success: false,
            error: "Contact not found"
          });
        }
        
        // Only allow the user who created the contact or admins to view it
        if (contact.userId && contact.userId.toString() !== req.userId) {
          // Check if user is admin (you may need to adjust this based on your admin check logic)
          const user = await User.findById(req.userId);
          const isAdmin = user && user.role === 'admin';
          
          if (!isAdmin) {
            return res.status(403).json({
              success: false,
              error: "You don't have permission to view this contact"
            });
          }
        }
        
        res.json({
          success: true,
          contact
        });
      } catch (error) {
        console.error("Error fetching contact:", error);
        res.status(500).json({
          success: false,
          error: "Failed to fetch contact"
        });
      }
} 
  
  module.exports = {
    submitContactForm,
    getAllContacts,
    getContactById,
    updateContactStatus,
    replyToContact,
    deleteContact,
    getContactStats,
    contactMessage
  };
  