const User = require('../Model/User_Model');
const Transaction = require('../Model/Transaction');
const Razorpay = require('razorpay');
const crypto = require('crypto');

// Initialize Razorpay with fallback for development
const razorpay = new Razorpay({
  key_id: process.env.RAZORPAY_KEY_ID || 'rzp_live_mAFHmvzbCdtMW3',
  key_secret: process.env.RAZORPAY_KEY_SECRET || '6ddrLjAT4C5L9Dk0QyXow18G'
});

// Define coin packages
const coinPackages = {
  basic: { coins: 50, price: 19 },
  standard: { coins: 200, price: 69 },
  premium: { coins: 500, price: 159 },
  ultimate: { coins: 1200, price: 349 },
  vip: { coins: 3000, price: 799 }
};

// Get user's coin balance
exports.getBalance = async (req, res) => {
  try {
    const user = await User.findById(req.user.id);
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }

    return res.status(200).json({
      success: true,
      coins: user.coins || 0,
      earnedCoins: user.earnedCoins || 0
    });
  } catch (error) {
    console.error('Error getting coin balance:', error);
    return res.status(500).json({ success: false, error: 'Server error' });
  }
};

// Create a Razorpay order
exports.createOrder = async (req, res) => {
  try {
    const { packageId, amount, coins } = req.body;
    
    // Validate package
    if (!coinPackages[packageId]) {
      return res.status(400).json({ success: false, error: 'Invalid package' });
    }
    
    // Verify amount matches package price
    const expectedAmount = coinPackages[packageId].price * 100; // in paise
    if (amount !== expectedAmount) {
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid amount',
        expected: expectedAmount,
        received: amount
      });
    }
    
    // Create a shorter receipt ID (must be <= 40 characters)
    // Use a shorter timestamp and only the last part of the user ID
    const shortUserId = req.user.id.substring(req.user.id.length - 8);
    const timestamp = Date.now().toString().substring(6); // Use only last 7 digits
    const receipt = `order_${shortUserId}_${timestamp}`;
    
    // Create Razorpay order
    const options = {
      amount: amount,
      currency: 'INR',
      receipt: receipt,
      notes: {
        userId: req.user.id,
        packageId: packageId,
        coins: coins
      }
    };
    
    const order = await razorpay.orders.create(options);
    
    return res.status(200).json({
      success: true,
      orderId: order.id,
      keyId: process.env.RAZORPAY_KEY_ID
    });
  } catch (error) {
    console.error('Error creating order:', error);
    return res.status(500).json({ 
      success: false, 
      error: 'Server error',
      details: error.message
    });
  }
};

// Verify payment and add coins to user
exports.verifyPayment = async (req, res) => {
  try {
    const { razorpay_order_id, razorpay_payment_id, razorpay_signature, packageId } = req.body;
    
    // Verify signature
    const body = razorpay_order_id + '|' + razorpay_payment_id;
    const expectedSignature = crypto
      .createHmac('sha256', process.env.RAZORPAY_KEY_SECRET)
      .update(body)
      .digest('hex');
    
    if (expectedSignature !== razorpay_signature) {
      return res.status(400).json({ success: false, error: 'Invalid signature' });
    }
    
    // Get order details
    const order = await razorpay.orders.fetch(razorpay_order_id);
    
    // Verify package
    if (!coinPackages[packageId]) {
      return res.status(400).json({ success: false, error: 'Invalid package' });
    }
    
    const coinsToAdd = coinPackages[packageId].coins;
    
    // Update user's coin balance
    const user = await User.findById(req.user.id);
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    user.coins = (user.coins || 0) + coinsToAdd;
    await user.save();
    
    // Create transaction record
    const transaction = new Transaction({
      user: req.user.id,
      amount: order.amount / 100, // Convert from paise to rupees
      coins: coinsToAdd,
      packageId: packageId,
      paymentId: razorpay_payment_id,
      orderId: razorpay_order_id,
      status: 'completed'
    });
    
    await transaction.save();
    
    return res.status(200).json({
      success: true,
      message: 'Payment verified and coins added',
      newBalance: user.coins
    });
  } catch (error) {
    console.error('Error verifying payment:', error);
    return res.status(500).json({ success: false, error: 'Server error' });
  }
};

// Function to grant initial coins to a new user
exports.grantInitialCoins = async (userId) => {
  try {
    // Find the user
    const user = await User.findById(userId);
    if (!user) {
      console.error(`User not found: ${userId}`);
      return false;
    }
    
    // Check if user already has the initialCoinGranted flag
    if (user.initialCoinGranted) {
      console.log(`User ${userId} already received initial coins`);
      return false;
    }
    
    // IMPORTANT: Set coins directly to 25 instead of incrementing
    // This ensures the user gets exactly 25 coins, not 25 + default value
    const result = await User.updateOne(
      { _id: userId, initialCoinGranted: { $ne: true } },
      { 
        $set: { 
          initialCoinGranted: true,
          coins: 25  // Changed from 20 to 25 coins
        }
      }
    );
    
    if (result.modifiedCount > 0) {
      console.log(`Set 25 initial coins to user ${userId}`);
      
      // Create a transaction record for the initial coins
      const transaction = new Transaction({
        user: userId,
        amount: 0, // Free coins
        coins: 25, // Changed from 20 to 25
        packageId: 'initial',
        paymentId: 'system_initial_grant',
        orderId: `initial_${Date.now()}`,
        status: 'completed',
        notes: 'Initial coins for new user'
      });
      
      await transaction.save();
      return true;
    }
    
    return false;
  } catch (error) {
    console.error('Error granting initial coins:', error);
    return false;
  }
};
