const mongoose = require("mongoose");

const replySchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  content: {
    type: String,
    required: true,
  },
  mentionedUser: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
  },
  likes: [
    {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
    },
  ],
  createdAt: {
    type: Date,
    default: Date.now,
  },
});

const reviewSchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  rating: {
    type: Number,
    required: true,
    min: 1,
    max: 5,
  },
  content: {
    type: String,
    required: true,
  },
  likes: [
    {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
    },
  ],
  replies: [replySchema],
  createdAt: {
    type: Date,
    default: Date.now,
  },
});

const readHistorySchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  lastRead: {
    type: Date,
    default: Date.now,
  },
  readCount: {
    type: Number,
    default: 1,
  },
});

const episodeSchema = new mongoose.Schema({
  title: { type: String, required: true },
  content: { type: String, required: true },
  chapterNumber: { type: Number },
  reads: { type: Number, default: 0 },
  likes: [{ type: mongoose.Schema.Types.ObjectId, ref: "User" }],
  reviews: [reviewSchema],
  reviewedUsers: [{ type: mongoose.Schema.Types.ObjectId, ref: "User" }],
  averageRating: { type: Number, default: 0 },
  updatedAt: { type: Date, default: Date.now },
  createdAt: { type: Date, default: Date.now },
  readers:[{
      type: mongoose.Schema.Types.ObjectId,
    ref: "User"
  }],
  totalReaders: {
    type: Number,
    default: 0
  },
  // Add this field to track unlocks
  unlocks: {
    type: Number,
    default: 0
  },
  coinEarnings: {
    type: Number,
    default: 0
  },
  readHistory: [readHistorySchema]
});

const seriesSchema = new mongoose.Schema(
  {
    title: { type: String, required: true },
    author: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    followers: [{
      type: mongoose.Schema.Types.ObjectId,
      ref: "User"
    }],
    coverImage: { type: String },
    summary: { type: String },
    storyType: { type: String, required: true },
    categories: [{ type: String }],
    status: {
      type: String,
      enum: ["ongoing", "completed", "draft"],
      default: "ongoing", // Change default from draft to ongoing
    },
    episodes: [episodeSchema],
    reads: { type: Number, default: 0 },
    readHistory: [readHistorySchema],
    likes: [{ type: mongoose.Schema.Types.ObjectId, ref: "User" }],
    reviews: [reviewSchema],
    reviewedUsers: [{ type: mongoose.Schema.Types.ObjectId, ref: "User" }],
    averageRating: { type: Number, default: 0 },
    uniqueReads: { type: Number, default: 0 },
    totalReads: { type: Number, default: 0 },
    language: {
      type: String,
      default: "EN",
      required: true,
    },
    // Add this field to track total coin earnings from this series
    totalCoinEarnings: {
      type: Number,
      default: 0
    }
  },
  { timestamps: true }
);

episodeSchema.pre('save', function(next) {
  if (this.readers) {
    this.totalReaders = this.readers.length;
  }
  
  // Calculate episode average rating
  if (this.reviews && this.reviews.length > 0) {
    const sum = this.reviews.reduce((acc, review) => acc + review.rating, 0);
    this.averageRating = (sum / this.reviews.length).toFixed(1);
  }
  
  next();
});

seriesSchema.pre("save", function (next) {
  // Calculate series average rating
  if (this.reviews.length > 0) {
    const sum = this.reviews.reduce((acc, review) => acc + review.rating, 0);
    this.averageRating = (sum / this.reviews.length).toFixed(1);
  }

  // Calculate episode average ratings
  this.episodes.forEach((episode) => {
    if (episode.reviews && episode.reviews.length > 0) {
      const sum = episode.reviews.reduce(
        (acc, review) => acc + review.rating,
        0
      );
      episode.averageRating = (sum / episode.reviews.length).toFixed(1);
    }
  });

  this.uniqueReads = this.readHistory.length;
  this.totalReads = this.readHistory.reduce(
    (sum, record) => sum + (record.readCount || 1),
    0
  );
  
  // Calculate total coin earnings from all episodes
  if (this.episodes && this.episodes.length > 0) {
    this.totalCoinEarnings = this.episodes.reduce(
      (sum, episode) => sum + (episode.coinEarnings || 0),
      0
    );
  }

  next();
});

// Add indexes for better query performance
seriesSchema.index({ author: 1 });
seriesSchema.index({ status: 1 });
seriesSchema.index({ categories: 1 });
seriesSchema.index({ "episodes._id": 1 });
seriesSchema.index({ language: 1 });
seriesSchema.index({ createdAt: -1 });
seriesSchema.index({ reads: -1 }, { name: 'reads_index' }); // Add this line here

seriesSchema.index({ reads: -1 });
seriesSchema.index({ likes: 1 });

const Series = mongoose.model("Series", seriesSchema);

module.exports = Series;
