const mongoose = require("mongoose");
const bcrypt = require("bcryptjs");
const crypto = require("crypto");

const userSchema = new mongoose.Schema(
  {
    username: {
      type: String,
      required: true,
      unique: true,
      trim: true,
    },
    email: {
      type: String,
      required: true,
      unique: true,
      trim: true,
      lowercase: true,
    },
    password: {
      type: String,
      required: function() {
        return !this.googleId; // Password required only if not using Google auth
      },
    },
    phoneNumber: {
      type: String,
      trim: true,
    },
    hasCompletedOnboarding: {
      type: Boolean,
      default: false,
    },
    googleId: {
      type: String,
    },
    profileImage: {
      type: String,
      default: "/uploads/default-profile.png",
    },
    coverImage: {
      type: String,
      default: "/uploads/default-cover.png",
    },
    bio: {
      type: String,
      default: "",
    },
    location: {
      type: String,
      default: "",
    },
    website: {
      type: String,
      default: "",
    },
    followers: [
      {
        type: mongoose.Schema.Types.ObjectId,
        ref: "User",
      },
    ],
    following: [
      {
        type: mongoose.Schema.Types.ObjectId,
        ref: "User",
      },
    ],
    // Add this to the userSchema, inside the main schema object
readerLevel: {
  type: String,
  enum: ['Bronze', 'Silver', 'Gold', 'Platinum'],
  default: 'Bronze'
},
coinsSpent: {
  type: Number,
  default: 0
},
badges: {
  bronze: {
    type: Boolean,
    default: true
  },
  silver: {
    type: Boolean,
    default: false
  },
  gold: {
    type: Boolean,
    default: false
  },
  platinum: {
    type: Boolean,
    default: false
  }
},

   // In the notifications array schema
notifications: [
  {
    type: {
      type: String,
      required: true
    },
    message: {
      type: String,
      required: true
    },
    fromUser: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      default: null
    },
    contentId: {
      type: mongoose.Schema.Types.ObjectId,
      required: true
    },
    contentModel: {
      type: String,
      required: true
    },
    createdAt: {
      type: Date,
      default: Date.now
    },
    isRead: {
      type: Boolean,
      default: false
    },
    paymentStatus: String,
    paymentAmount: Number,
    additionalData: {
      type: mongoose.Schema.Types.Mixed,
      default: {}
    }
  }
]
,
    library: {
      stories: [
        {
          type: mongoose.Schema.Types.ObjectId,
          ref: "Story",
        },
      ],
      series: [
        {
          type: mongoose.Schema.Types.ObjectId,
          ref: "Series",
        },
      ],
    },
    writingStats: {
      totalWordCount: {
        type: Number,
        default: 0,
      },
      totalEarnings: {
        type: Number,
        default: 0,
      },
      dailyWordCount: {
        type: Map,
        of: {
          count: Number,
          earnings: Number,
        },
        default: {},
      },
      contentWordCounts: [
        {
          contentId: mongoose.Schema.Types.ObjectId,
          contentType: {
            type: String,
            enum: ["story", "episode"],
          },
          wordCount: Number,
          earnings: Number,
          date: {
            type: Date,
            default: Date.now,
          },
        },
      ],
    },
    // Account details for bank transfers and redemptions
    accountDetails: {
      bankName: {
        type: String,
        trim: true
      },
      accountNumber: {
        type: String,
        trim: true
      },
      ifscCode: {
        type: String,
        trim: true
      },
      accountHolderName: {
        type: String,
        trim: true
      },
      panNumber: {
        type: String,
        trim: true
      },
      mobileNumber: {
        type: String,
        trim: true
      }
    },
    // Coin system
    coins: {
      type: Number,
      default: 0, 
    },
    // Add this field to your User model schema
initialCoinGranted: {
  type: Boolean,
  default: false
}
,
    earnedCoins: {
      type: Number,
      default: 0,
    },
    coinTransactions: [
      {
        amount: Number,
        from: {
          type: mongoose.Schema.Types.ObjectId,
          ref: "User",
        },
        fromUsername: String,
        to: {
          type: mongoose.Schema.Types.ObjectId,
          ref: "User",
        },
        seriesId: {
          type: mongoose.Schema.Types.ObjectId,
          ref: "Series",
        },
        seriesTitle: String,
        episodeId: String,
        episodeTitle: String,
        transactionType: {
          type: String,
          enum: ["purchase", "tip", "reward", "redeem"],
        },
        transactionDate: {
          type: Date,
          default: Date.now,
        },
      },
    ],
    coinRedemptions: [
      {
        amount: Number,
        value: Number, // Value in rupees after platform fee
        fullValue: Number, // Full value before platform fee
        platformFee: Number, // Platform fee amount
        status: {
          type: String,
          enum: ["pending", "approved", "rejected"],
          default: "pending",
        },
        requestedAt: {
          type: Date,
          default: Date.now,
        },
        processedAt: Date,
        notes: String,
      },
    ],
// In the User schema
unlockedEpisodes: {
  type: Object,
  default: {}
}
,
    resetPasswordToken: String,
    resetPasswordExpire: Date,
// From User_Model.js
isAdmin: {
  type: Boolean,
  default: false,
},
lastActive: {
  type: Date,
  default: Date.now
},

role: {
  type: String,
  enum: ["user", "writer", "admin", "moderator"],
  default: "user"
},


  },
  { timestamps: true }
);

// Index for faster queries
// userSchema.index({ username: 1 });
// userSchema.index({ email: 1 });
userSchema.index({ googleId: 1 });
userSchema.index({ "notifications.isRead": 1 });
userSchema.index({ "library.stories": 1 });
userSchema.index({ "library.series": 1 });

// Pre-save hook to hash password
userSchema.pre("save", async function (next) {
  if (!this.isModified("password") || !this.password) {
    return next();
  }
  try {
    const salt = await bcrypt.genSalt(10);
    this.password = await bcrypt.hash(this.password, salt);
    next();
  } catch (error) {
    next(error);
  }
});

// Method to update word count stats
userSchema.methods.updateWordCount = async function (contentId, contentType, content) {
  const wordCount = content.split(/\s+/).filter(word => word.length > 0).length;
  const earnings = wordCount * 0.10;
  const today = new Date().toISOString().split("T")[0];
  
  // Initialize daily word count if it doesn't exist
  if (!this.writingStats.dailyWordCount.get(today)) {
    this.writingStats.dailyWordCount.set(today, { count: 0, earnings: 0 });
  }
  
  // Update daily word count
  const dailyStats = this.writingStats.dailyWordCount.get(today);
  dailyStats.count += wordCount;
  dailyStats.earnings += earnings;
  this.writingStats.dailyWordCount.set(today, dailyStats);
  
  // Update total word count and earnings
  this.writingStats.totalWordCount += wordCount;
  this.writingStats.totalEarnings += earnings;
  
  // Add to content word counts
  this.writingStats.contentWordCounts.push({
    contentId,
    contentType,
    wordCount,
    earnings,
    date: new Date()
  });
  
  await this.save();
  return { wordCount, earnings };
};

// Method to compare password
userSchema.methods.comparePassword = async function (candidatePassword) {
  return await bcrypt.compare(candidatePassword, this.password);
};

// Method to get reset password token
userSchema.methods.getResetPasswordToken = function () {
  // Generate token
  const resetToken = crypto.randomBytes(20).toString("hex");

  // Hash and set to resetPasswordToken field
  this.resetPasswordToken = crypto
    .createHash("sha256")
    .update(resetToken)
    .digest("hex");

  // Set expire
  this.resetPasswordExpire = Date.now() + 10 * 60 * 1000;

  return resetToken;
};

// Method to check if an episode is unlocked
userSchema.methods.isEpisodeUnlocked = function(seriesId, episodeId) {
  if (!this.unlockedEpisodes) return false;
  
  const seriesIdStr = seriesId.toString();
  const unlockedEpisodes = this.unlockedEpisodes.get(seriesIdStr) || [];
  
  return unlockedEpisodes.includes(episodeId.toString());
};

// Method to unlock an episode
userSchema.methods.unlockEpisode = function(seriesId, episodeId) {
  if (!this.unlockedEpisodes) {
    this.unlockedEpisodes = new Map();
  }
  
  const seriesIdStr = seriesId.toString();
  const episodeIdStr = episodeId.toString();
  
  // Get current unlocked episodes for this series
  const unlockedEpisodes = this.unlockedEpisodes.get(seriesIdStr) || [];
  
  // Add the episode if it's not already unlocked
  if (!unlockedEpisodes.includes(episodeIdStr)) {
    unlockedEpisodes.push(episodeIdStr);
    this.unlockedEpisodes.set(seriesIdStr, unlockedEpisodes);
    return true; // Episode was newly unlocked
  }
  
  return false; // Episode was already unlocked
};

// Helper method to check if account details are complete
userSchema.methods.hasCompleteAccountDetails = function() {
  if (!this.accountDetails) return false;
  
  const requiredFields = ['bankName', 'accountNumber', 'ifscCode', 'accountHolderName', 'panNumber', 'mobileNumber'];
  
  return requiredFields.every(field => 
    this.accountDetails[field] && this.accountDetails[field].trim() !== ''
  );
};

// Helper method to get missing account detail fields
userSchema.methods.getMissingAccountFields = function() {
  if (!this.accountDetails) return ['bankName', 'accountNumber', 'ifscCode', 'accountHolderName', 'panNumber', 'mobileNumber'];
  
  const requiredFields = ['bankName', 'accountNumber', 'ifscCode', 'accountHolderName', 'panNumber', 'mobileNumber'];
  
  return requiredFields.filter(field => 
    !this.accountDetails[field] || this.accountDetails[field].trim() === ''
  );
};

const User = mongoose.model("User", userSchema);

module.exports = User;
