const express = require("express");
const { OAuth2Client } = require("google-auth-library");
const crypto = require("crypto");
const client = new OAuth2Client(process.env.GOOGLE_CLIENT_ID);
const axios = require("axios");
const router = express.Router();
const multer = require("multer");
const fs = require("fs");
const path = require("path");
const NodeCache = require("node-cache");
const UserController = require("../Controller/User_Controller");
const { auth, adminAuth, adminImpersonationAuth } = require("../middleware/auth");
const activityTracker = require("../middleware/activityTracker");

// Initialize cache with different TTL for different types of data
const staticCache = new NodeCache({ stdTTL: 1800, checkperiod: 600 }); // 30 minutes for static data
const dynamicCache = new NodeCache({ stdTTL: 300, checkperiod: 120 }); // 5 minutes for dynamic data
const userCache = new NodeCache({ stdTTL: 60, checkperiod: 30 }); // 1 minute for user-specific data

const User = require("../Model/User_Model");
const storyController = require("../Controller/Story_Controller");
const seriesController = require("../Controller/Series_Controller");
const CategoryController = require("../Controller/Category_Controller");
const LibraryController = require("../Controller/Library_Controller");
const SearchController = require("../Controller/Search_Controller");
const AdminController = require("../Controller/Admin/Admin_Controller");
const AdminLoginController = require("../Controller/Admin/AdminLogin_Controller");
const WriterContestManagmentController = require("../Controller/Admin/Writer_Contest_Managment_Controller");
const categoryUpload = require("../middleware/categoryUpload");
const CategoryAdminController = require("../Controller/Admin/Category_admin_Controller");
const coinController = require("../Controller/coinController");
const ContactController = require("../Controller/Contact_Controller");

// Import model of series and story 
const Story = require("../Model/Story_Model");
const Series = require("../Model/Series_Model");

// Cache types enum
const CACHE_TYPES = {
  STATIC: 'static',     // Categories, contest info - rarely changes
  DYNAMIC: 'dynamic',   // Story lists, series lists - changes moderately
  USER: 'user',         // User-specific data - changes frequently
  REALTIME: 'realtime'  // Never cache - likes, views, coins, etc.
};

// Define what should be cached and what shouldn't
const CACHE_RULES = {
  // NEVER cache these - they need real-time updates
  NEVER_CACHE: [
    'like', 'increment-reads', 'increment-views', 'tip', 'unlock',
    'coins', 'payment', 'reviews', 'library', 'notifications',
    'follow', 'reader', 'lock-status', 'episode.*lock', 'episodes.*lock'
  ],
  
  // Cache with user context (short TTL)
  USER_SPECIFIC: [
    'profile', 'stories', 'series', 'library', 'notifications'
  ],
  
  // Cache globally (medium TTL)
  DYNAMIC_CONTENT: [
    'published-stories', 'published-series', 'trending-series', 'all-stories'
  ],
  
  // Cache globally (long TTL)
  STATIC_CONTENT: [
    'categories', 'contests/active', 'stats'
  ]
};

// Smart cache key generator
const generateCacheKey = (req, cacheType = CACHE_TYPES.DYNAMIC) => {
  const baseUrl = req.originalUrl || req.url;
  const urlParts = baseUrl.split('?');
  const path = urlParts[0];
  const query = urlParts[1] || '';
  
  // For user-specific cache, include user ID
  if (cacheType === CACHE_TYPES.USER && req.userId) {
    return `user:${req.userId}:${path}:${query}`;
  }
  
  // For dynamic content, include query params
  if (cacheType === CACHE_TYPES.DYNAMIC) {
    return `dynamic:${path}:${query}`;
  }
  
  // For static content, just use path
  return `static:${path}`;
};

// Enhanced function to check if route should NEVER be cached
const isNeverCacheRoute = (path) => {
  const neverCachePatterns = [
    /\/lock-status/,
    /\/unlock/,
    /\/coins/,
    /\/payment/,
    /\/like/,
    /\/increment-reads/,
    /\/increment-views/,
    /\/tip/,
    /\/reviews/,
    /\/library/,
    /\/notifications/,
    /\/follow/,
    /\/reader/
  ];
  
  return neverCachePatterns.some(pattern => pattern.test(path));
};

// Determine if route should be cached
const shouldCache = (req) => {
  const path = req.originalUrl || req.url;
  
  // Check if it's a never-cache route using patterns
  if (isNeverCacheRoute(path)) {
    return { cache: false, reason: `Route matches never-cache pattern` };
  }
  
  // Never cache routes with these keywords
  for (const keyword of CACHE_RULES.NEVER_CACHE) {
    if (path.includes(keyword)) {
      return { cache: false, reason: `Contains never-cache keyword: ${keyword}` };
    }
  }
  
  // Don't cache POST, PUT, DELETE requests
  if (['POST', 'PUT', 'DELETE'].includes(req.method)) {
    return { cache: false, reason: 'Mutating HTTP method' };
  }
  
  // Check for bypass flag
  if (req.bypassCache) {
    return { cache: false, reason: 'Cache bypass flag set' };
  }
  
  // Don't cache authenticated requests unless specifically allowed
  if (req.userId && !isAllowedAuthenticatedCache(path)) {
    return { cache: false, reason: 'Authenticated request not in whitelist' };
  }
  
  return { cache: true };
};

// Check if authenticated request can be cached
const isAllowedAuthenticatedCache = (path) => {
  const allowedPatterns = [
    '/categories', '/stats', '/contests/active', '/published-stories',
    '/published-series', '/trending-series', '/all-stories'
  ];
  
  return allowedPatterns.some(pattern => path.includes(pattern));
};

// Get appropriate cache instance
const getCacheInstance = (cacheType) => {
  switch (cacheType) {
    case CACHE_TYPES.STATIC:
      return staticCache;
    case CACHE_TYPES.USER:
      return userCache;
    case CACHE_TYPES.DYNAMIC:
    default:
      return dynamicCache;
  }
};

// Smart cache middleware with enhanced bypass logic
const smartCacheMiddleware = (cacheType = CACHE_TYPES.DYNAMIC, customTTL = null) => {
  return (req, res, next) => {
    // Force bypass for lock-related routes
    if (isNeverCacheRoute(req.originalUrl || req.url)) {
      console.log(`🚫 NEVER CACHE: ${req.originalUrl} - Lock/Payment/Interaction route`);
      return next();
    }
    
    // Check if cache should be bypassed
    if (req.bypassCache) {
      console.log(`🚫 Cache bypassed for ${req.originalUrl}: bypassCache flag set`);
      return next();
    }
    
    const cacheCheck = shouldCache(req);
    
    if (!cacheCheck.cache) {
      console.log(`🚫 Skipping cache for ${req.originalUrl}: ${cacheCheck.reason}`);
      return next();
    }
    
    const cacheKey = generateCacheKey(req, cacheType);
    const cacheInstance = getCacheInstance(cacheType);
    const cachedResponse = cacheInstance.get(cacheKey);
    
    if (cachedResponse) {
      console.log(`✅ Cache hit for ${cacheKey}`);
      return res.json(cachedResponse);
    }
    
    console.log(`❌ Cache miss for ${cacheKey}`);
    
    // Store the original res.json function
    const originalJson = res.json;
    
    // Override res.json method to cache the response
    res.json = function(data) {
      // Only cache successful responses and if not bypassed
      if (data && (data.success === undefined || data.success === true) && !req.bypassCache) {
        const ttl = customTTL || cacheInstance.options.stdTTL;
        cacheInstance.set(cacheKey, data, ttl);
        console.log(`💾 Cached response for ${cacheKey} with TTL ${ttl}s`);
      }
      
      // Call the original json method
      return originalJson.call(this, data);
    };
    
    next();
  };
};

// Comprehensive lock status cache invalidation
const invalidateLockStatusCache = (seriesId, episodeId = null, userId = null) => {
  console.log(`🧹 Starting lock status cache invalidation for series: ${seriesId}, episode: ${episodeId}, user: ${userId}`);
  
  const patterns = [
    // Clear ALL lock status related cache
    /lock-status/,
    /unlock/,
    new RegExp(`series/${seriesId}`),
    new RegExp(`episode/${episodeId || '.*'}`),
  ];
  
  if (userId) {
    patterns.push(new RegExp(`user:${userId}`));
  }
  
  let totalInvalidated = 0;
  
  // Apply patterns to all cache instances
  [staticCache, dynamicCache, userCache].forEach((cache, index) => {
    const cacheNames = ['static', 'dynamic', 'user'];
    const keys = cache.keys();
    
    patterns.forEach(pattern => {
      const matchingKeys = keys.filter(key => pattern.test(key));
      matchingKeys.forEach(key => {
        console.log(`🗑️ Invalidating ${cacheNames[index]} cache: ${key}`);
        cache.del(key);
        totalInvalidated++;
      });
    });
  });
  
  console.log(`✅ Lock status cache invalidation complete. Invalidated ${totalInvalidated} entries`);
  return totalInvalidated;
};

// Smart cache invalidation
const smartInvalidateCache = (contentType, contentId = null, userId = null) => {
  const patterns = [];
  
  switch (contentType) {
    case 'story':
      patterns.push(
        /dynamic:\/published-stories/,
        /dynamic:\/all-stories/,
        /dynamic:\/stories\/category/
      );
      if (contentId) {
        patterns.push(new RegExp(`\/stories\/${contentId}`));
      }
      if (userId) {
        patterns.push(new RegExp(`user:${userId}:\/stories`));
      }
      break;
      
    case 'series':
      patterns.push(
        /dynamic:\/published-series/,
        /dynamic:\/trending-series/,
        /dynamic:\/series-by-language/,
        /dynamic:\/series\/category/
      );
      if (contentId) {
        patterns.push(new RegExp(`\/series\/${contentId}`));
      }
      if (userId) {
        patterns.push(new RegExp(`user:${userId}:\/series`));
      }
      break;
      
    case 'episode':
      if (contentId) {
        patterns.push(new RegExp(`\/series\/${contentId}`));
        // Also invalidate any lock status cache for this series
        invalidateLockStatusCache(contentId, null, userId);
      }
      break;
      
    case 'category':
      staticCache.flushAll(); // Categories change rarely, flush all static cache
      patterns.push(/dynamic:\/content\/category/);
      break;
      
    case 'user':
      if (userId) {
        // Clear all user-specific cache
        const userKeys = userCache.keys().filter(key => key.startsWith(`user:${userId}:`));
        userKeys.forEach(key => userCache.del(key));
      }
      break;
  }
  
  // Apply patterns to appropriate cache instances
  [staticCache, dynamicCache, userCache].forEach(cache => {
    const keys = cache.keys();
    patterns.forEach(pattern => {
      const matchingKeys = keys.filter(key => pattern.test(key));
      matchingKeys.forEach(key => {
        console.log(`🗑️ Invalidating cache: ${key}`);
        cache.del(key);
      });
    });
  });
};

// Middleware for cache invalidation after mutations
const invalidationMiddleware = (contentType, options = {}) => {
  return (req, res, next) => {
    const originalEnd = res.end;
    
    res.end = function(chunk, encoding) {
      if (res.statusCode >= 200 && res.statusCode < 300) {
        const contentId = req.params.id || req.params.storyId || req.params.seriesId || options.contentId;
        const userId = req.userId;
        
        smartInvalidateCache(contentType, contentId, userId);
      }
      
      return originalEnd.call(this, chunk, encoding);
    };
    
    next();
  };
};

// Configure multer for file uploads (unchanged)
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = "uploads";
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  },
});

const fileFilter = (req, file, cb) => {
  if (file.mimetype.startsWith("image/")) {
    cb(null, true);
  } else {
    cb(null, false);
    cb(new Error("Only image files are allowed"));
  }
};

const upload = multer({
  storage: storage,
  fileFilter: fileFilter,
  limits: { fileSize: 10 * 1024 * 1024 },
});

// ===== AUTHENTICATION ROUTES (No caching) =====
router.post("/signup", UserController.signup);
router.post("/signin", UserController.signin);
router.post("/google-auth", UserController.googleAuth);

// ===== USER PROFILE ROUTES (User-specific caching) =====
router.get("/profile", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.USER, 60), UserController.getProfile);
router.put("/profile/avatar", auth, activityTracker, invalidationMiddleware('user'), upload.single("profileImage"), UserController.updateProfileImage);
router.put("/profile/cover", auth, activityTracker, invalidationMiddleware('user'), upload.single("coverImage"), UserController.updateCoverImage);
router.put("/profile/details", auth, activityTracker, invalidationMiddleware('user'), UserController.updateProfileDetails);
router.put("/profile/account-details", auth, activityTracker, invalidationMiddleware('user'), UserController.updateAccountDetails);
router.post("/users/:userId/follow", auth, activityTracker, invalidationMiddleware('user'), UserController.followUser);
router.get("/stats", smartCacheMiddleware(CACHE_TYPES.STATIC, 600), UserController.getStats);
router.get('/profile/basic', auth, UserController.getBasicProfile);
router.post("/update-phone", auth, invalidationMiddleware('user'), UserController.updatePhoneNumber);
router.get("/onboarding-status", auth, UserController.checkOnboardingStatus);

// ===== USER SOCIAL ROUTES (No caching - real-time data) =====
router.get("/users/:userId/followers", auth, activityTracker, UserController.getFollowers);
router.get("/users/:userId/following", auth, activityTracker, UserController.getFollowing);
router.get("/users/:userId/check-follow", auth, activityTracker, UserController.checkFollowStatus);

router.get('/profile/stories', auth, smartCacheMiddleware(CACHE_TYPES.USER, 120), UserController.getUserStories);
router.get('/profile/series', auth, smartCacheMiddleware(CACHE_TYPES.USER, 120), UserController.getUserSeries);
router.get('/profile/stats', auth, UserController.getUserStats);
router.get('/users/:userId/stories', auth, smartCacheMiddleware(CACHE_TYPES.USER, 120), UserController.getUserStories);
router.get('/users/:userId/series', auth, smartCacheMiddleware(CACHE_TYPES.USER, 120), UserController.getUserSeries);
router.get('/users/:userId/stats', auth, UserController.getUserStats);
router.get("/users/:userId/profile", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.USER, 300), UserController.getUserProfile);

// ===== NOTIFICATION ROUTES (No caching - real-time) =====
router.get("/notifications", auth, activityTracker, UserController.getNotifications);
router.put("/notifications/mark-read", auth, activityTracker, UserController.markNotificationsAsRead);
router.post("/notifications/like", auth, activityTracker, UserController.createLikeNotification);
router.post("/notifications/review", auth, activityTracker, UserController.createReviewNotification);
router.post("/notifications/reply", auth, activityTracker, UserController.createReplyNotification);
router.delete("/notifications/:id", auth, activityTracker, UserController.deleteNotification);

// ===== STORY ROUTES =====
// Cached routes (public content)
router.get("/published-stories", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), storyController.getAllPublishedStories);
router.get("/all-stories", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), storyController.getAllStories);

// User-specific routes (no cache or short cache)
router.get("/stories", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.USER, 60), storyController.getUserStories);
router.get("/story-drafts", auth, activityTracker, storyController.getStoryDrafts);
router.get("/stories/:id", auth, activityTracker, storyController.getStoryById);

// Mutation routes (no cache, with invalidation)
router.post("/stories", auth, activityTracker, invalidationMiddleware('story'), storyController.createStory);
router.put("/stories/:id", auth, activityTracker, invalidationMiddleware('story'), storyController.updateStory);
router.delete("/stories/:id", auth, activityTracker, invalidationMiddleware('story'), storyController.deleteStory);
router.delete("/story-drafts/:id", auth, activityTracker, invalidationMiddleware('story'), storyController.deleteStoryDraft);

// Real-time interaction routes (NEVER cache)
router.post("/stories/:id/increment-reads", auth, activityTracker, storyController.incrementReads);
router.post("/stories/:id/like", auth, activityTracker, storyController.likeStory);
router.post("/stories/:id/tip", auth, activityTracker, storyController.tipAuthor);

// ===== STORY REVIEW ROUTES (No caching - real-time interactions) =====
router.get("/stories/:id/reviews", auth, activityTracker, storyController.getStoryReviews);
router.post("/stories/:id/reviews", auth, activityTracker, storyController.addReview);
router.put("/stories/:id/reviews/:reviewId", auth, activityTracker, storyController.updateReview);
router.delete("/stories/:id/reviews/:reviewId", auth, activityTracker, storyController.deleteReview);
router.post("/stories/:id/reviews/:reviewId/like", auth, activityTracker, storyController.likeReview);
router.post("/stories/:id/reviews/:reviewId/reply", auth, activityTracker, storyController.addReplyToReview);
router.get("/stories/:id/reviews/:reviewId", auth, activityTracker, storyController.getReviewWithReplies);
router.put("/stories/:id/reviews/:reviewId/replies/:replyId", auth, activityTracker, storyController.updateReply);
router.delete("/stories/:id/reviews/:reviewId/replies/:replyId", auth, activityTracker, storyController.deleteReply);
router.post("/stories/:id/reviews/:reviewId/replies/:replyId/like", auth, activityTracker, storyController.likeReply);
router.post("/stories/:id/reviews/:reviewId/replies/:replyId/reply", auth, activityTracker, storyController.addNestedReply);

// ===== SERIES ROUTES =====
// Cached public routes
router.get("/series-by-language", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), async (req, res) => {
  try {
    await seriesController.getAllPublishedSeries(req, res);
  } catch (error) {
    console.error("Error in series-by-language route:", error);
    res.status(500).json({ 
      success: false, 
      error: "An error occurred while fetching series by language" 
    });
  }
});

router.get("/published-series", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), async (req, res) => {
  try {
    const timeoutPromise = new Promise((_, reject) => 
      setTimeout(() => reject(new Error('Request timeout')), 5000)
    );
    
    const controllerPromise = seriesController.getAllPublishedSeries(req, res);
    
    await Promise.race([controllerPromise, timeoutPromise])
      .catch(err => {
        console.error("Error in published-series route:", err);
        res.json({
          success: true,
          series: [],
          message: "Error fetching series, please try again later"
        });
      });
  } catch (error) {
    console.error("Error in published-series route:", error);
    res.status(200).json({ 
      success: true, 
      series: [],
      message: "Error fetching series, please try again later"
    });
  }
});

router.get("/trending-series", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 180), async (req, res) => {
  try {
    const timeoutPromise = new Promise((_, reject) => 
      setTimeout(() => reject(new Error('Request timeout')), 5000)
    );
    
    const controllerPromise = seriesController.getTrendingSeries(req, res);
    
    await Promise.race([controllerPromise, timeoutPromise])
      .catch(err => {
        console.error("Error in trending-series route:", err);
        res.json({
          success: true,
          series: [],
          message: "Error fetching trending series, please try again later"
        });
      });
  } catch (error) {
    console.error("Error in trending-series route:", error);
    res.status(200).json({ 
      success: true, 
      series: [],
      message: "Error fetching trending series, please try again later"
    });
  }
});

// User-specific routes
router.get("/series", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.USER, 60), seriesController.getUserSeries);
router.get("/series-drafts", auth, activityTracker, seriesController.getDrafts);
router.get("/series/:id", auth, activityTracker, seriesController.getSeriesById);

// Mutation routes
router.post("/series", auth, activityTracker, invalidationMiddleware('series'), seriesController.createSeries);
router.put("/series/:id/details", auth, activityTracker, invalidationMiddleware('series'), seriesController.updateSeriesDetails);
router.delete("/series/:id", auth, activityTracker, invalidationMiddleware('series'), seriesController.deleteSeries);
router.delete("/series-drafts/:id", auth, activityTracker, invalidationMiddleware('series'), seriesController.deleteSeriesDraft);
router.put("/series/:id/status", auth, activityTracker, invalidationMiddleware('series'), seriesController.updateSeriesStatus);
router.post("/api/series/:id/upload-cover", auth, activityTracker, invalidationMiddleware('series'), upload.single("coverImage"), seriesController.uploadCoverImage);

// Real-time interaction routes (NEVER cache)
router.post("/series/:id/like", auth, activityTracker, seriesController.likeSeries);
router.post("/series/:id/increment-reads", auth, activityTracker, seriesController.incrementReads);

// ===== EPISODE ROUTES =====
// Content routes (no cache for individual episodes)
router.get("/series/:id/episode/:episodeId", auth, activityTracker, seriesController.getEpisodeById);
router.get("/series/:id/episode/:episodeId/basic-info", auth, activityTracker, seriesController.getBasicEpisodeInfo);

// Mutation routes
router.post("/series/:id/episode", auth, activityTracker, invalidationMiddleware('episode'), seriesController.addEpisode);
router.put("/series/:id/episode/:episodeId", auth, activityTracker, invalidationMiddleware('episode'), seriesController.updateEpisode);
router.delete("/series/:id/episode/:episodeId", auth, activityTracker, invalidationMiddleware('episode'), seriesController.deleteEpisode);

// Real-time interaction routes (NEVER cache)
router.post("/series/:id/episode/:episodeId/like", auth, activityTracker, seriesController.likeEpisode);
router.post("/series/:id/episode/:episodeId/increment-reads", auth, activityTracker, seriesController.incrementEpisodeReads);
router.post('/series/:id/episode/:episodeId/reader', auth, activityTracker, async (req, res) => {
  try {
    await seriesController.addEpisodeReader(req, res);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// ===== EPISODE REVIEW ROUTES (No caching - real-time interactions) =====
router.get("/series/:id/episode/:episodeId/reviews", auth, activityTracker, seriesController.getEpisodeReviews);
router.post("/series/:id/episode/:episodeId/reviews", auth, activityTracker, seriesController.addEpisodeReview);
router.put("/series/:id/episode/:episodeId/reviews/:reviewId", auth, activityTracker, seriesController.updateEpisodeReview);
router.delete("/series/:id/episode/:episodeId/reviews/:reviewId", auth, activityTracker, seriesController.deleteEpisodeReview);
router.post("/series/:id/episode/:episodeId/reviews/:reviewId/like", auth, activityTracker, seriesController.likeEpisodeReview);
router.post("/series/:id/episode/:episodeId/reviews/:reviewId/reply", auth, activityTracker, seriesController.addEpisodeReviewReply);
router.post("/series/:id/episode/:episodeId/reviews/:reviewId/replies/:replyId/like", auth, activityTracker, seriesController.likeEpisodeReply);
router.put("/series/:id/episode/:episodeId/reviews/:reviewId/replies/:replyId", auth, activityTracker, seriesController.updateEpisodeReply);
router.delete("/series/:id/episode/:episodeId/reviews/:reviewId/replies/:replyId", auth, activityTracker, seriesController.deleteEpisodeReply);

// ===== CRITICAL: EPISODE LOCK/UNLOCK ROUTES (NEVER CACHE - REAL-TIME REQUIRED) =====

// 🔥 IMMEDIATE UNLOCK ROUTE - WITH COMPREHENSIVE CACHE CLEARING
router.post("/series/:id/episode/:episodeId/unlock", auth, activityTracker, async (req, res, next) => {
  console.log(`🔓 UNLOCK REQUEST: Series ${req.params.id}, Episode ${req.params.episodeId}, User ${req.userId}`);
  
  // Store original response methods
  const originalJson = res.json;
  const originalEnd = res.end;
  
  // Override response to ensure immediate cache invalidation
  res.json = function(data) {
    console.log(`📤 UNLOCK RESPONSE: ${JSON.stringify(data)}`);
    
    // If unlock was successful, immediately clear ALL related cache
    if (data && data.success) {
      console.log(`✅ UNLOCK SUCCESSFUL - Starting immediate cache invalidation`);
      
      // Comprehensive cache invalidation
      const seriesId = req.params.id;
      const episodeId = req.params.episodeId;
      const userId = req.userId;
      
      // 1. Clear lock status cache immediately
      invalidateLockStatusCache(seriesId, episodeId, userId);
      
      // 2. Clear series cache
      smartInvalidateCache('episode', seriesId, userId);
      
      // 3. Clear any user-specific cache
      const userKeys = userCache.keys().filter(key => 
        key.includes(`user:${userId}`) && 
        (key.includes(`series/${seriesId}`) || key.includes(`episode/${episodeId}`))
      );
      userKeys.forEach(key => {
        console.log(`🗑️ Clearing user cache: ${key}`);
        userCache.del(key);
      });
      
      // 4. Clear dynamic cache for this series
      const dynamicKeys = dynamicCache.keys().filter(key => 
        key.includes(`series/${seriesId}`) || key.includes(`episode/${episodeId}`)
      );
      dynamicKeys.forEach(key => {
        console.log(`🗑️ Clearing dynamic cache: ${key}`);
        dynamicCache.del(key);
      });
      
      console.log(`🧹 CACHE CLEARED - Unlock cache invalidation complete`);
    }
    
    // Call original json method
    return originalJson.call(this, data);
  };
  
  next();
}, seriesController.unlockEpisode);

// 🔥 LOCK STATUS ROUTES - NEVER CACHE, ALWAYS REAL-TIME
router.get("/series/:id/episode/:episodeId/lock-status", auth, activityTracker, (req, res, next) => {
  console.log(`🔍 LOCK STATUS CHECK: Series ${req.params.id}, Episode ${req.params.episodeId}, User ${req.userId}`);
  
  // Force real-time data - never use cache
  req.bypassCache = true;
  
  // Add headers to prevent any client-side caching
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0',
    'Surrogate-Control': 'no-store'
  });
  
  next();
}, seriesController.checkEpisodeLock);

router.get("/series/:id/episodes/lock-status", auth, activityTracker, (req, res, next) => {
  console.log(`🔍 SERIES LOCK STATUS CHECK: Series ${req.params.id}, User ${req.userId}`);
  
  // Force real-time data - never use cache
  req.bypassCache = true;
  
  // Add headers to prevent any client-side caching
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0',
    'Surrogate-Control': 'no-store'
  });
  
  next();
}, seriesController.checkSeriesEpisodesLockStatus);

router.get('/series/:id/episode/:episodeId/lock-status', auth, (req, res, next) => {
  console.log(`🔍 EPISODE LOCK STATUS: Series ${req.params.id}, Episode ${req.params.episodeId}, User ${req.userId}`);
  
  // Force real-time data - never use cache
  req.bypassCache = true;
  
  // Add headers to prevent any client-side caching
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0',
    'Surrogate-Control': 'no-store'
  });
  
  next();
}, seriesController.getEpisodeLockStatus);

// ===== LIBRARY ROUTES (No caching - user-specific real-time data) =====
router.post("/library/:type/:id", auth, activityTracker, LibraryController.addToLibrary);
router.get("/library", auth, activityTracker, LibraryController.getLibrary);

// ===== CATEGORY ROUTES (Static caching - changes rarely) =====
router.get("/content/category/:category", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 600), CategoryController.getCategoryContent);
router.get("/categories", smartCacheMiddleware(CACHE_TYPES.STATIC, 1800), CategoryController.getActiveCategories);
router.get("/trending-contest-content", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 600), CategoryController.getTrendingContestContent);
router.get("/stories/category/:category", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), storyController.getStoriesByCategory);
router.get("/series/category/:category", auth, activityTracker, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), seriesController.getSeriesByCategory);
router.get("/categories/home", smartCacheMiddleware(CACHE_TYPES.STATIC, 1800), CategoryController.getCategoriesForHome);
router.get("/content-by-category", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 600), CategoryController.getContentByCategory);

// ===== SEARCH ROUTES (No caching - dynamic results) =====
router.get("/search", auth, activityTracker, SearchController.searchContent);

// ===== COIN SYSTEM ROUTES (NEVER cache - financial transactions) =====
router.get("/coins/earnings", auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0'
  });
  next();
}, seriesController.getCoinEarnings);

router.get("/coins/balance", auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0'
  });
  next();
}, UserController.getCoinBalance);

router.post("/coins/redeem", auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  next();
}, UserController.redeemCoins);

router.get("/coins/redemption-history", auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0'
  });
  next();
}, UserController.getRedemptionHistory);

router.get("/coins/earnings/detailed", auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0'
  });
  next();
}, seriesController.getDetailedCoinEarnings);

// ===== PAYMENT ROUTES (NEVER cache - financial transactions) =====
router.get('/coins/balance', auth, activityTracker, (req, res, next) => {
  req.bypassCache = true;
  res.set({
    'Cache-Control': 'no-store, no-cache, must-revalidate',
    'Pragma': 'no-cache',
    'Expires': '0'
  });
  next();
}, coinController.getBalance);

router.post('/coins/create-order', auth, activityTracker, coinController.createOrder);
router.post('/coins/verify-payment', auth, activityTracker, coinController.verifyPayment);

// ===== CONTEST ROUTES =====
router.get("/api/contests/active", smartCacheMiddleware(CACHE_TYPES.STATIC, 600), WriterContestManagmentController.getActiveContests);
router.post("/api/contests/:contestId/join", auth, activityTracker, invalidationMiddleware('contest'), WriterContestManagmentController.joinContest);
router.get("/api/contests/joined", auth, activityTracker, WriterContestManagmentController.getJoinedContests);
router.put("/api/admin/contests/:contestId/set-featured-leaderboard", adminAuth, invalidationMiddleware('contest'), WriterContestManagmentController.setFeaturedLeaderboard);
router.get("/api/contests/:contestId/status", auth, WriterContestManagmentController.checkContestJoinStatus);

// Contest leaderboard routes
router.get("/api/admin/contests/:contestId/leaderboard", adminAuth, WriterContestManagmentController.getContestLeaderboard);
router.post("/api/admin/contests/featured-leaderboard", adminAuth, invalidationMiddleware('contest'), WriterContestManagmentController.setFeaturedLeaderboard);
router.get("/api/contests/featured-leaderboard", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 600), WriterContestManagmentController.getFeaturedLeaderboard);

// ===== ADMIN AUTHENTICATION ROUTES (No caching) =====
router.post("/api/admin/login", AdminLoginController.adminLogin);
router.post("/api/admin/register", AdminLoginController.registerAdmin);
router.get('/admin/check-user', adminAuth, AdminLoginController.checkUserExists);
router.post('/admin/reset-user-password', adminAuth, AdminLoginController.adminResetUserPassword);

// ===== ADMIN DASHBOARD ROUTES =====
router.get("/api/admin/writers", adminAuth, async (req, res, next) => {
  try {
    await AdminController.getAllWriters(req, res);
  } catch (error) {
    next(error);
  }
});
router.get("/api/admin/stats", adminAuth, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), AdminController.getAdminStats);
router.get("/api/admin/enhanced-stats", adminAuth, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), AdminController.getEnhancedAdminStats);
router.get("/api/admin/writer/:id/stats", adminAuth, AdminController.getWriterStats);
router.get("/api/admin/writer/:id/account", adminAuth, AdminController.getWriterAccountDetails);
router.get("/api/admin/writer/:id/payments", adminAuth, AdminController.getWriterPayments);
router.get("/api/admin/activity-trend", adminAuth, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 600), AdminController.getActivityTrend);

// ===== ADMIN WRITER MANAGEMENT ROUTES =====
router.get('/api/admin/writer/:writerId', adminAuth, AdminController.getWriterDetails);
router.get('/api/admin/writer/:writerId/stories', adminAuth, AdminController.getWriterStories);
router.get('/api/admin/writer/:writerId/series', adminAuth, AdminController.getWriterSeries);
router.get('/api/admin/impersonate/:writerId', adminImpersonationAuth, UserController.adminImpersonateUser);

router.get('/writer/:writerId', adminAuth, AdminController.getWriterDetails);
router.get('/writer/:writerId/stories', adminAuth, AdminController.getWriterStories);
router.get('/writer/:writerId/series', adminAuth, AdminController.getWriterSeries);

// ===== ADMIN PAYMENT MANAGEMENT ROUTES (No caching - financial data) =====
router.post("/api/admin/writer/payment", adminAuth, AdminController.recordWriterPayment);
router.put("/api/admin/writer/payment", adminAuth, AdminController.updatePayment);
router.delete("/api/admin/writer/:writerId/payment/:paymentId", adminAuth, AdminController.deletePayment);
router.get("/admin/redemption-requests", adminAuth, UserController.getRedemptionRequests);
router.post("/admin/process-redemption", adminAuth, UserController.processRedemption);

// ===== ADMIN CONTENT MANAGEMENT ROUTES =====
router.put('/api/admin/story/:storyId', adminAuth, invalidationMiddleware('story'), AdminController.updateStory);
router.put('/api/admin/series/:seriesId', adminAuth, invalidationMiddleware('series'), AdminController.updateSeries);
router.delete('/api/admin/story/:storyId', adminAuth, invalidationMiddleware('story'), AdminController.deleteStory);
router.delete('/api/admin/series/:seriesId', adminAuth, invalidationMiddleware('series'), AdminController.deleteSeries);

// ===== ADMIN CATEGORY MANAGEMENT ROUTES =====
router.get("/api/admin/categories", adminAuth, CategoryAdminController.getAllCategories);
router.post("/api/admin/categories", adminAuth, invalidationMiddleware('category'), categoryUpload.single("coverImage"), CategoryAdminController.addCategory);
router.put("/api/admin/categories/:id", adminAuth, invalidationMiddleware('category'), categoryUpload.single("coverImage"), CategoryAdminController.updateCategory);
router.delete("/api/admin/categories/:id", adminAuth, invalidationMiddleware('category'), CategoryAdminController.deleteCategory);
router.patch("/api/admin/categories/:id/toggle-status", adminAuth, invalidationMiddleware('category'), CategoryAdminController.toggleCategoryStatus);
router.patch("/api/admin/categories/:id/toggle-home", adminAuth, invalidationMiddleware('category'), CategoryAdminController.toggleCategoryHomeStatus);
router.post("/api/admin/categories/update-order", adminAuth, invalidationMiddleware('category'), CategoryAdminController.updateCategoryOrder);

// ===== ADMIN CONTEST MANAGEMENT ROUTES =====
router.post("/api/admin/contests", adminAuth, invalidationMiddleware('contest'), WriterContestManagmentController.createContest);
router.get("/api/admin/contests", adminAuth, WriterContestManagmentController.getAllContests);
router.put("/api/admin/contests/status", adminAuth, invalidationMiddleware('contest'), WriterContestManagmentController.updateContestStatus);
router.delete("/api/admin/contests/:contestId", auth, invalidationMiddleware('contest'), WriterContestManagmentController.deleteContest);
router.put("/api/admin/contests/:contestId", adminAuth, invalidationMiddleware('contest'), WriterContestManagmentController.updateContest);
router.get("/api/admin/contests/:contestId/submissions", adminAuth, WriterContestManagmentController.getContestSubmissions);

// ===== CONTACT FORM ROUTES =====
router.post("/contact", ContactController.submitContactForm);
router.get("/admin/contacts", adminAuth, ContactController.getAllContacts);
router.get("/admin/contacts/stats", adminAuth, smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 300), ContactController.getContactStats);
router.get("/admin/contacts/:id", adminAuth, ContactController.getContactById);
router.put("/admin/contacts/:id", adminAuth, ContactController.updateContactStatus);
router.post("/admin/contacts/:id/reply", adminAuth, ContactController.replyToContact);
router.delete("/admin/contacts/:id", adminAuth, ContactController.deleteContact);
router.get('/contact/:id', auth, ContactController.contactMessage);

// ===== ADMIN COIN MANAGEMENT ROUTES (No caching - financial operations) =====
router.post('/admin/grant-initial-coins', auth, adminAuth, UserController.grantInitialCoinsToAllUsers);

// ===== READER LEVEL ROUTES =====
router.get('/reader/level', auth, UserController.getUserReaderLevelData);
router.get("/reader/wall-of-fame", smartCacheMiddleware(CACHE_TYPES.DYNAMIC, 1800), UserController.getReaderWallOfFame);
router.post("/reader/update-coins-spent", auth, activityTracker, UserController.updateCoinsSpent);
router.get('/users/:userId/reader-level', auth, UserController.getUserReaderLevel);

// ===== ENHANCED CACHE MANAGEMENT ROUTES =====

// Get comprehensive cache statistics
router.get('/admin/cache-analytics', adminAuth, (req, res) => {
  try {
    const staticStats = staticCache.getStats();
    const dynamicStats = dynamicCache.getStats();
    const userStats = userCache.getStats();
    
    const staticKeys = staticCache.keys();
    const dynamicKeys = dynamicCache.keys();
    const userKeys = userCache.keys();
    
    // Calculate hit rates
    const calculateHitRate = (stats) => {
      const total = stats.hits + stats.misses;
      return total > 0 ? ((stats.hits / total) * 100).toFixed(2) : 0;
    };
    
    // Get memory usage estimation
    const getMemoryUsage = (cache, keys) => {
      let totalSize = 0;
      keys.forEach(key => {
        const value = cache.get(key);
        if (value) {
          totalSize += JSON.stringify(value).length;
        }
      });
      return (totalSize / 1024 / 1024).toFixed(2); // MB
    };
    
    res.json({
      success: true,
      timestamp: new Date().toISOString(),
      cacheAnalytics: {
        static: {
          stats: staticStats,
          hitRate: `${calculateHitRate(staticStats)}%`,
          keyCount: staticKeys.length,
          memoryUsage: `${getMemoryUsage(staticCache, staticKeys)} MB`,
          ttl: staticCache.options.stdTTL
        },
        dynamic: {
          stats: dynamicStats,
          hitRate: `${calculateHitRate(dynamicStats)}%`,
          keyCount: dynamicKeys.length,
          memoryUsage: `${getMemoryUsage(dynamicCache, dynamicKeys)} MB`,
          ttl: dynamicCache.options.stdTTL
        },
        user: {
          stats: userStats,
          hitRate: `${calculateHitRate(userStats)}%`,
          keyCount: userKeys.length,
          memoryUsage: `${getMemoryUsage(userCache, userKeys)} MB`,
          ttl: userCache.options.stdTTL
        }
      },
      systemMemory: {
        rss: `${(process.memoryUsage().rss / 1024 / 1024).toFixed(2)} MB`,
        heapUsed: `${(process.memoryUsage().heapUsed / 1024 / 1024).toFixed(2)} MB`,
        heapTotal: `${(process.memoryUsage().heapTotal / 1024 / 1024).toFixed(2)} MB`
      }
    });
  } catch (error) {
    console.error("Error getting cache analytics:", error);
    res.status(500).json({
      success: false,
      error: "Failed to get cache analytics"
    });
  }
});

// Smart cache invalidation endpoint
router.post('/admin/smart-invalidate', adminAuth, (req, res) => {
  try {
    const { type, contentId, userId, pattern } = req.body;
    
    if (pattern) {
      // Custom pattern invalidation
      const regex = new RegExp(pattern);
      let totalInvalidated = 0;
      
      [staticCache, dynamicCache, userCache].forEach(cache => {
        const keys = cache.keys();
        const matchingKeys = keys.filter(key => regex.test(key));
        matchingKeys.forEach(key => cache.del(key));
        totalInvalidated += matchingKeys.length;
      });
      
      return res.json({
        success: true,
        message: `Invalidated ${totalInvalidated} cache entries matching pattern: ${pattern}`
      });
    }
    
    if (type) {
      smartInvalidateCache(type, contentId, userId);
      return res.json({
        success: true,
        message: `Smart invalidation completed for type: ${type}`,
        contentId,
        userId
      });
    }
    
    res.status(400).json({
      success: false,
      error: "Either 'type' or 'pattern' is required"
    });
  } catch (error) {
    console.error("Error in smart invalidation:", error);
    res.status(500).json({
      success: false,
      error: "Failed to perform smart invalidation"
    });
  }
});

// Cache warming endpoint
router.post('/admin/warm-cache', adminAuth, async (req, res) => {
  try {
    const { endpoints } = req.body;
    
    if (!endpoints || !Array.isArray(endpoints)) {
      return res.status(400).json({
        success: false,
        error: "Endpoints array is required"
      });
    }
    
    const results = [];
    
    for (const endpoint of endpoints) {
      try {
        // Create a mock request to warm the cache
        const mockReq = {
          originalUrl: endpoint,
          url: endpoint,
          method: 'GET',
          query: {},
          userId: null
        };
        
        const cacheKey = generateCacheKey(mockReq, CACHE_TYPES.DYNAMIC);
        
        // Check if already cached
        if (dynamicCache.get(cacheKey)) {
          results.push({
            endpoint,
            status: 'already_cached',
            cacheKey
          });
          continue;
        }
        
        // Here you would typically make an internal request to the endpoint
        // For now, we'll just mark it as attempted
        results.push({
          endpoint,
          status: 'warming_attempted',
          cacheKey
        });
        
      } catch (error) {
        results.push({
          endpoint,
          status: 'error',
          error: error.message
        });
      }
    }
    
    res.json({
      success: true,
      message: "Cache warming completed",
      results
    });
  } catch (error) {
    console.error("Error warming cache:", error);
    res.status(500).json({
      success: false,
      error: "Failed to warm cache"
    });
  }
});

// Cache key inspection
router.get('/admin/cache-keys/:type?', adminAuth, (req, res) => {
  try {
    const { type } = req.params;
    const { search, limit = 50 } = req.query;
    
    let cache, cacheName;
    
    switch (type) {
      case 'static':
        cache = staticCache;
        cacheName = 'Static Cache';
        break;
      case 'dynamic':
        cache = dynamicCache;
        cacheName = 'Dynamic Cache';
        break;
      case 'user':
        cache = userCache;
        cacheName = 'User Cache';
        break;
      default:
        // Return all caches
        const allKeys = {
          static: staticCache.keys().slice(0, limit),
          dynamic: dynamicCache.keys().slice(0, limit),
          user: userCache.keys().slice(0, limit)
        };
        
        return res.json({
          success: true,
          allCacheKeys: allKeys,
          totalKeys: {
            static: staticCache.keys().length,
            dynamic: dynamicCache.keys().length,
            user: userCache.keys().length
          }
        });
    }
    
    let keys = cache.keys();
    
    if (search) {
      keys = keys.filter(key => key.toLowerCase().includes(search.toLowerCase()));
    }
    
    const limitedKeys = keys.slice(0, parseInt(limit));
    const keyDetails = limitedKeys.map(key => {
      const ttl = cache.getTtl(key);
      const value = cache.get(key);
      
      return {
        key,
        ttl,
        expiresAt: ttl ? new Date(ttl).toISOString() : null,
        size: value ? JSON.stringify(value).length : 0,
        hasValue: !!value
      };
    });
    
    res.json({
      success: true,
      cacheName,
      totalKeys: keys.length,
      displayedKeys: limitedKeys.length,
      keys: keyDetails
    });
  } catch (error) {
    console.error("Error inspecting cache keys:", error);
    res.status(500).json({
      success: false,
      error: "Failed to inspect cache keys"
    });
  }
});

// Selective cache clear
router.delete('/admin/cache/:type', adminAuth, (req, res) => {
  try {
    const { type } = req.params;
    const { keys } = req.body;
    
    let cache, cacheName;
    
    switch (type) {
      case 'static':
        cache = staticCache;
        cacheName = 'Static Cache';
        break;
      case 'dynamic':
        cache = dynamicCache;
        cacheName = 'Dynamic Cache';
        break;
      case 'user':
        cache = userCache;
        cacheName = 'User Cache';
        break;
      case 'all':
        staticCache.flushAll();
        dynamicCache.flushAll();
        userCache.flushAll();
        return res.json({
          success: true,
          message: "All caches cleared successfully"
        });
      default:
        return res.status(400).json({
          success: false,
          error: "Invalid cache type. Use 'static', 'dynamic', 'user', or 'all'"
        });
    }
    
    if (keys && Array.isArray(keys)) {
      // Clear specific keys
      let clearedCount = 0;
      keys.forEach(key => {
        if (cache.del(key)) {
          clearedCount++;
        }
      });
      
      res.json({
        success: true,
        message: `Cleared ${clearedCount} keys from ${cacheName}`,
        clearedKeys: keys
      });
    } else {
      // Clear entire cache type
      cache.flushAll();
      res.json({
        success: true,
        message: `${cacheName} cleared successfully`
      });
    }
  } catch (error) {
    console.error("Error clearing cache:", error);
    res.status(500).json({
      success: false,
      error: "Failed to clear cache"
    });
  }
});

// Cache health check
router.get('/admin/cache-health', adminAuth, (req, res) => {
  try {
    const checkCacheHealth = (cache, name) => {
      const stats = cache.getStats();
      const keys = cache.keys();
      const hitRate = stats.hits + stats.misses > 0 ? 
        ((stats.hits / (stats.hits + stats.misses)) * 100).toFixed(2) : 0;
      
      return {
        name,
        status: keys.length > 0 ? 'active' : 'empty',
        keyCount: keys.length,
        hitRate: `${hitRate}%`,
        hits: stats.hits,
        misses: stats.misses,
        keptItems: stats.keptItems,
        varyAccept: stats.varyAccept
      };
    };
    
    const health = {
      static: checkCacheHealth(staticCache, 'Static Cache'),
      dynamic: checkCacheHealth(dynamicCache, 'Dynamic Cache'),
      user: checkCacheHealth(userCache, 'User Cache')
    };
    
    // Overall health assessment
    const totalKeys = health.static.keyCount + health.dynamic.keyCount + health.user.keyCount;
    const overallStatus = totalKeys > 0 ? 'healthy' : 'no_cache';
    
    res.json({
      success: true,
      timestamp: new Date().toISOString(),
      overallStatus,
      totalCachedItems: totalKeys,
      cacheHealth: health,
      recommendations: generateCacheRecommendations(health)
    });
  } catch (error) {
    console.error("Error checking cache health:", error);
    res.status(500).json({
      success: false,
      error: "Failed to check cache health"
    });
  }
});

// Generate cache recommendations
const generateCacheRecommendations = (health) => {
  const recommendations = [];
  
  // Check hit rates
  Object.values(health).forEach(cache => {
    const hitRate = parseFloat(cache.hitRate);
    if (hitRate < 50 && cache.keyCount > 10) {
      recommendations.push(`${cache.name} has low hit rate (${cache.hitRate}). Consider reviewing cache strategy.`);
    }
  });
  
  // Check if user cache is too large
  if (health.user.keyCount > 1000) {
    recommendations.push("User cache has many entries. Consider reducing TTL or implementing cleanup.");
  }
  
  // Check if static cache is empty
  if (health.static.keyCount === 0) {
    recommendations.push("Static cache is empty. Consider pre-warming frequently accessed data.");
  }
  
  if (recommendations.length === 0) {
    recommendations.push("Cache performance looks good!");
  }
  
  return recommendations;
};

// ===== EMERGENCY CACHE CLEAR FOR UNLOCK ISSUES =====
router.post('/admin/emergency-unlock-cache-clear', adminAuth, (req, res) => {
  try {
    const { seriesId, episodeId, userId } = req.body;
    
    console.log(`🚨 EMERGENCY CACHE CLEAR: Series ${seriesId}, Episode ${episodeId}, User ${userId}`);
    
    // Clear ALL cache related to unlock/lock status
    const patterns = [
      /lock-status/,
      /unlock/,
      /coins/,
      /episode/,
      /series/
    ];
    
    if (seriesId) {
      patterns.push(new RegExp(`series/${seriesId}`));
    }
    
    if (episodeId) {
      patterns.push(new RegExp(`episode/${episodeId}`));
    }
    
    if (userId) {
      patterns.push(new RegExp(`user:${userId}`));
    }
    
    let totalCleared = 0;
    
    [staticCache, dynamicCache, userCache].forEach((cache, index) => {
      const cacheNames = ['static', 'dynamic', 'user'];
      const keys = cache.keys();
      
      patterns.forEach(pattern => {
        const matchingKeys = keys.filter(key => pattern.test(key));
        matchingKeys.forEach(key => {
          console.log(`🗑️ Emergency clearing ${cacheNames[index]}: ${key}`);
          cache.del(key);
          totalCleared++;
        });
      });
    });
    
    res.json({
      success: true,
      message: `Emergency cache clear completed. Cleared ${totalCleared} entries.`,
      clearedCount: totalCleared,
      seriesId,
      episodeId,
      userId
    });
  } catch (error) {
    console.error("Error in emergency cache clear:", error);
    res.status(500).json({
      success: false,
      error: "Failed to perform emergency cache clear"
    });
  }
});

// ===== DEBUG ROUTES =====
router.get("/debug/notifications", auth, async (req, res) => {
  try {
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    const notificationTypes = [...new Set(user.notifications.map(n => n.type))];
    const notificationCounts = {};
    notificationTypes.forEach(type => {
      notificationCounts[type] = user.notifications.filter(n => n.type === type).length;
    });
    
    res.json({
      success: true,
      totalNotifications: user.notifications.length,
      notificationTypes,
      notificationCounts,
      recentNotifications: user.notifications.slice(-5)
    });
  } catch (error) {
    console.error("Error debugging notifications:", error);
    res.status(500).json({ error: error.message });
  }
});

router.post("/debug/test-level-up", auth, async (req, res) => {
  try {
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    
    const testNotification = {
      type: "level_up",
      message: "Congratulations! You've reached Gold Reader level and earned 100 bonus coins!",
      fromUser: null,
      contentId: user._id,
      contentModel: "User",
      createdAt: new Date(),
      isRead: false,
      additionalData: {
        previousLevel: "Silver",
        newLevel: "Gold",
        bonusCoins: 100
      }
    };
    
    user.notifications.push(testNotification);
    await user.save();
    
    res.json({
      success: true,
      message: "Test level-up notification created",
      notification: testNotification
    });
  } catch (error) {
    console.error("Error creating test notification:", error);
    res.status(500).json({ error: error.message });
  }
});

router.get("/debug/user-data", auth, async (req, res) => {
  try {
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.json({
      success: true,
      userId: req.userId,
      username: user.username,
      earnedCoins: user.earnedCoins,
      coinRedemptions: user.coinRedemptions,
      hasRedemptionsField: user.hasOwnProperty('coinRedemptions'),
      isRedemptionsArray: Array.isArray(user.coinRedemptions),
      accountDetails: user.accountDetails,
      hasAccountDetails: !!user.accountDetails,
      accountDetailsKeys: user.accountDetails ? Object.keys(user.accountDetails) : [],
      schemaFields: Object.keys(user._doc)
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

router.post("/debug/fix-account", auth, async (req, res) => {
  try {
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    await User.updateOne(
      { _id: req.userId },
      { 
        $set: { 
          accountDetails: {
            bankName: "Test Bank",
            accountNumber: "1234567890",
            ifscCode: "ABCD0123456",
            accountHolderName: "Test User",
            panNumber: "ABCDE1234F",
            mobileNumber: "9876543210"
          }
        }
      }
    );
    const updatedUser = await User.findById(req.userId);
    
    res.json({
      success: true,
      message: "Account details fixed",
      hasAccountDetails: !!updatedUser.accountDetails,
      accountDetails: updatedUser.accountDetails
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

router.get("/api/contests/:contestId/status", async (req, res) => {
  try {
    const { contestId } = req.params;
    const contest = await Contest.findById(contestId);
    
    if (!contest) {
      return res.status(404).json({ success: false, message: "Contest not found" });
    }
    
    res.json({
      success: true,
      status: contest.status,
      isActive: contest.status === "active"
    });
  } catch (error) {
    console.error("Error fetching contest status:", error);
    res.status(500).json({ success: false, message: "Server error" });
  }
});

router.post("/debug/fix-account-details", auth, async (req, res) => {
  try {
    const {
      bankName = "Test Bank",
      accountNumber = "1234567890",
      ifscCode = "ABCD0123456",
      accountHolderName = "Test User",
      panNumber = "ABCDE1234F",
      mobileNumber = "9876543210"
    } = req.body;
    
    const result = await User.updateOne(
      { _id: req.userId },
      {
        $set: {
          accountDetails: {
            bankName,
            accountNumber,
            ifscCode,
            accountHolderName,
            panNumber,
            mobileNumber
          }
        }
      }
    );
    
    const updatedUser = await User.findById(req.userId);
    
    res.json({
      success: true,
      message: "Account details fixed",
      updateResult: result,
      accountDetails: updatedUser.accountDetails,
      hasAccountDetails: !!updatedUser.accountDetails
    });
  } catch (error) {
    console.error("Error fixing account details:", error);
    res.status(500).json({ error: error.message });
  }
});

router.get('/debug/notifications/:userId', adminAuth, async (req, res) => {
  try {
    const { userId } = req.params;
    const user = await User.findById(userId);
    
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    const contactReplies = user.notifications.filter(n => n.type === "contact_reply");
    
    res.json({
      success: true,
      userId,
      totalNotifications: user.notifications.length,
      contactReplyNotifications: contactReplies,
      allNotificationTypes: [...new Set(user.notifications.map(n => n.type))]
    });
  } catch (error) {
    console.error("Error fetching notifications:", error);
    res.status(500).json({ error: error.message });
  }
});

router.post('/admin/fix-contact-notifications', adminAuth, async (req, res) => {
  try {
    const { userId, contactId, subject, reply } = req.body;
    
    if (!userId || !contactId || !subject || !reply) {
      return res.status(400).json({ 
        success: false, 
        error: "Missing required fields" 
      });
    }
    
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ 
        success: false, 
        error: "User not found" 
      });
    }
    
    const notificationIndex = user.notifications.findIndex(
      n => n.type === 'contact_reply' && 
      n.contentId.toString() === contactId
    );
    
    if (notificationIndex === -1) {
      return res.status(404).json({ 
        success: false, 
        error: "Notification not found" 
      });
    }
    
    user.notifications[notificationIndex].additionalData = {
      reply: reply,
      originalSubject: subject
    };
    
    await user.save();
    
    res.json({
      success: true,
      message: "Notification updated successfully",
      notification: user.notifications[notificationIndex]
    });
  } catch (error) {
    console.error("Error fixing notification:", error);
    res.status(500).json({ 
      success: false, 
      error: "Failed to fix notification" 
    });
  }
});

router.get('/contact/:id', auth, async (req, res) => {
  try {
    const contact = await Contact.findById(req.params.id);
    
    if (!contact) {
      return res.status(404).json({
        success: false,
        error: "Contact not found"
      });
    }
    
    if (contact.userId && contact.userId.toString() !== req.userId) {
      return res.status(403).json({
        success: false,
        error: "You don't have permission to view this contact"
      });
    }
    
    res.json({
      success: true,
      contact
    });
  } catch (error) {
    console.error("Error fetching contact:", error);
    res.status(500).json({
      success: false,
      error: "Failed to fetch contact"
    });
  }
});

// ===== UNLOCK DEBUG ROUTES =====
router.get('/debug/unlock-status/:seriesId/:episodeId', auth, async (req, res) => {
  try {
    const { seriesId, episodeId } = req.params;
    const userId = req.userId;
    
    console.log(`🔍 DEBUG UNLOCK STATUS: Series ${seriesId}, Episode ${episodeId}, User ${userId}`);
    
    // Check cache status
    const cacheKeys = [
      ...staticCache.keys().filter(key => key.includes(seriesId) || key.includes(episodeId)),
      ...dynamicCache.keys().filter(key => key.includes(seriesId) || key.includes(episodeId)),
      ...userCache.keys().filter(key => key.includes(userId) && (key.includes(seriesId) || key.includes(episodeId)))
    ];
    
    // Get actual database status (you'll need to implement this in your controller)
    // This is just a placeholder - replace with actual database check
    const dbStatus = {
      message: "Replace this with actual database check from your Series controller"
    };
    
    res.json({
      success: true,
      debug: {
        seriesId,
        episodeId,
        userId,
        timestamp: new Date().toISOString(),
        cacheKeys,
        cacheKeyCount: cacheKeys.length,
        dbStatus,
        recommendation: cacheKeys.length > 0 ? 
          "Cache found - may be causing stale data" : 
          "No cache found - should show real-time data"
      }
    });
  } catch (error) {
    console.error("Error in unlock debug:", error);
    res.status(500).json({ error: error.message });
  }
});

router.post('/debug/force-unlock-cache-clear/:seriesId/:episodeId', auth, async (req, res) => {
  try {
    const { seriesId, episodeId } = req.params;
    const userId = req.userId;
    
    console.log(`🧹 FORCE UNLOCK CACHE CLEAR: Series ${seriesId}, Episode ${episodeId}, User ${userId}`);
    
    // Force clear all related cache
    const clearedCount = invalidateLockStatusCache(seriesId, episodeId, userId);
    
    res.json({
      success: true,
      message: `Force cleared ${clearedCount} cache entries`,
      seriesId,
      episodeId,
      userId,
      clearedCount,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error("Error in force cache clear:", error);
    res.status(500).json({ error: error.message });
  }
});

// ===== HEALTH CHECK ROUTE =====
router.get('/health', (req, res) => {
  res.status(200).json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    cache: {
      static: {
        keys: staticCache.keys().length,
        stats: staticCache.getStats()
      },
      dynamic: {
        keys: dynamicCache.keys().length,
        stats: dynamicCache.getStats()
      },
      user: {
        keys: userCache.keys().length,
        stats: userCache.getStats()
      }
    },
    unlockCachePolicy: {
      lockStatusRoutes: "NEVER_CACHED",
      unlockRoute: "IMMEDIATE_INVALIDATION",
      coinRoutes: "NEVER_CACHED",
      cacheBypassHeaders: "ENABLED"
    }
  });
});

// ===== ERROR HANDLING MIDDLEWARE =====
router.use((err, req, res, next) => {
  console.error(`API Error: ${req.method} ${req.url}`, err);
  res.status(500).json({
    error: "An unexpected error occurred",
    message: process.env.NODE_ENV === 'development' ? err.message : 'Internal server error'
  });
});

module.exports = router;

